<?php

namespace App\Http\Controllers\Api\Panel;

use App\Http\Controllers\Api\Controller;
use App\Http\Resources\CertificateResource;
use App\Mixins\Certificate\MakeCertificate;
use App\Models\Api\Certificate;
use App\Models\CertificateTemplate;
use App\Models\Api\Quiz;
use App\Models\Api\QuizzesResult;
use App\Models\Reward;
use App\Models\RewardAccounting;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Intervention\Image\Facades\Image;

class CertificatesController extends Controller
{
    public function created(Request $request)
    {
        $user = apiAuth();

        $quizzes = Quiz::where('creator_id', $user->id)
            ->where('status', Quiz::ACTIVE)->handleFilters()->get();

        return apiResponse2(1, 'retrieved', trans('public.retrieved'), [
            'certificates' => CertificateResource::collection($quizzes),
        ]);


    }

    public function students()
    {
        $user = apiAuth();

        $quizzes = Quiz::where('creator_id', $user->id)
            ->pluck('id')->toArray();


        $ee = Certificate::whereIn('quiz_id', $quizzes)
            ->get()
            ->map(function ($certificate) {

                return $certificate->details;

            });

        return apiResponse2(1, 'retrieved', trans('public.retrieved'), $ee);
    }

    public function achievements(Request $request)
    {
        $user = apiAuth();
        $results = QuizzesResult::where('user_id', $user->id)->where('status', QuizzesResult::$passed)
            ->whereHas('quiz', function ($query) {
                $query->where('status', Quiz::ACTIVE);
            })
            ->get()->map(function ($result) {

                return array_merge($result->details,
                    ['certificate' => $result->certificate->brief ?? null]
                );

            });


        return apiResponse2(1, 'retrieved', trans('public.retrieved'), $results);


    }

    public function makeCertificate($quizResultId)
    {
        $user = apiAuth();

        $makeCertificate = new MakeCertificate();

        $quizResult = QuizzesResult::where('id', $quizResultId)
            ->where('user_id', $user->id)
            ->where('status', QuizzesResult::$passed)
            ->with(['quiz' => function ($query) {
                $query->with(['webinar']);
            }])
            ->first();

        if (!empty($quizResult)) {
            return $makeCertificate->makeQuizCertificate($quizResult);
        }

        abort(404);
    }

    /**
     * Get certificate HTML content for mobile app
     */
    public function getCertificateHtml($certificateId)
    {
        $user = apiAuth();
        
        // Find the certificate
        $certificate = Certificate::where('id', $certificateId)
            ->where('student_id', $user->id)
            ->first();
            
        if (!$certificate) {
            return response()->json(['error' => 'Certificate not found'], 404);
        }
        
        // Get the appropriate template based on certificate type
        $template = CertificateTemplate::where('status', 'publish')
            ->where('type', $certificate->type)
            ->first();
            
        if (!$template) {
            return response()->json(['error' => 'Certificate template not found'], 404);
        }
        
        // Generate the certificate body
        $makeCertificate = new MakeCertificate();
        
        $body = '';
        $courseTitle = '';
        $teacherName = '';
        $duration = '';
        
        if ($certificate->type === 'quiz' && $certificate->quiz) {
            $quiz = $certificate->quiz;
            $body = $makeCertificate->makeBody(
                $template,
                $certificate,
                $user,
                $template->body,
                $quiz->webinar ? $quiz->webinar->title : '-',
                $certificate->user_grade,
                $quiz->webinar ? $quiz->webinar->teacher->id : null,
                $quiz->webinar ? $quiz->webinar->teacher->full_name : null,
                $quiz->webinar ? $quiz->webinar->duration : null
            );
        } elseif ($certificate->type === 'course' && $certificate->webinar) {
            $course = $certificate->webinar;
            $locale = app()->getLocale();
            $templateBody = (!empty($template->translate($locale)) and !empty($template->translate($locale)->body)) 
                ? $template->translate($locale)->body 
                : $template->body;
                
            $body = $makeCertificate->makeBody(
                $template,
                $certificate,
                $user,
                $templateBody,
                $course->title,
                null,
                $course->teacher->id,
                $course->teacher->full_name,
                $course->duration
            );
        } elseif ($certificate->type === 'bundle' && $certificate->bundle) {
            $bundle = $certificate->bundle;
            $locale = app()->getLocale();
            $templateBody = (!empty($template->translate($locale)) and !empty($template->translate($locale)->body)) 
                ? $template->translate($locale)->body 
                : $template->body;
                
            $body = $makeCertificate->makeBody(
                $template,
                $certificate,
                $user,
                $templateBody,
                $bundle->title,
                null,
                $bundle->teacher->id,
                $bundle->teacher->full_name,
                $bundle->duration
            );
        }
        
        $data = [
            'body' => $body
        ];
        
        // Generate the HTML using the same template as the backend
        $html = (string)view()->make('admin.certificates.create_template.show_certificate', $data);
        
        return response($html, 200, [
            'Content-Type' => 'text/html; charset=utf-8'
        ]);
    }

}

