<?php

namespace App\Mixins\Certificate;

use App\Models\Certificate;
use App\Models\CertificateTemplate;
use App\Models\UserMeta;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Facades\Image;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Barryvdh\DomPDF\Facade\Pdf;

class MakeCertificate
{
    public function makeQuizCertificate($quizResult)
    {
        $template = CertificateTemplate::where('status', 'publish')
            ->where('type', 'quiz')
            ->first();

        if (!empty($template)) {
            $quiz = $quizResult->quiz;
            $user = $quizResult->user;

            $userCertificate = $this->saveQuizCertificate($user, $quiz, $quizResult);

            $body = $this->makeBody(
                $template,
                $userCertificate,
                $user,
                $template->body,
                $quiz->webinar ? $quiz->webinar->title : '-',
                $quizResult->user_grade,
                $quiz->webinar->teacher->id,
                $quiz->webinar->teacher->full_name,
                $quiz->webinar->duration);

            $data = [
                'body' => $body
            ];

            $html = (string)view()->make('admin.certificates.create_template.show_certificate', $data);
            return $this->generateCertificateImage($userCertificate, $html);
        }

        abort(404);
    }

    public function saveQuizCertificate($user, $quiz, $quizResult)
    {
        $certificate = Certificate::where('quiz_id', $quiz->id)
            ->where('student_id', $user->id)
            ->where('quiz_result_id', $quizResult->id)
            ->first();

        $data = [
            'quiz_id' => $quiz->id,
            'student_id' => $user->id,
            'quiz_result_id' => $quizResult->id,
            'user_grade' => $quizResult->user_grade,
            'type' => 'quiz',
            'created_at' => $quizResult->created_at,
        ];

        if (!empty($certificate)) {
            $certificate->update($data);
        } else {
            $certificate = Certificate::create($data);

            $notifyOptions = [
                '[c.title]' => $quiz->webinar ? $quiz->webinar->title : '-',
            ];
            sendNotification('new_certificate', $notifyOptions, $user->id);
        }

        return $certificate;
    }

    public function makeBody($template, $userCertificate, $user, $body, $courseTitle = null, $userGrade = null, $teacherId = null, $teacherFullName = null, $duration = null)
    {
        $platformName = getGeneralSettings("site_name");

        $body = str_replace('[student]', $user->full_name, $body);
        $body = str_replace('[student_name]', $user->full_name, $body);
        $body = str_replace('[platform_name]', $platformName, $body);
        $body = str_replace('[course]', $courseTitle, $body);
        $body = str_replace('[course_name]', $courseTitle, $body);
        $body = str_replace('[grade]', $userGrade, $body);
        $body = str_replace('[certificate_id]', $userCertificate->id, $body);
        $body = str_replace('[date]', dateTimeFormat($userCertificate->created_at, 'j M Y | H:i'), $body);
        $body = str_replace('[instructor_name]', $teacherFullName, $body);
        $body = str_replace('[duration]', $duration, $body);

        $qrCode = $this->makeQrCode($template);

        if (!empty($qrCode)) {
            $body = str_replace('[qr_code]', $qrCode, $body);
        }

        $instructorSignatureImg = null;
        if (!empty($teacherId)) {
            $instructorSignature = UserMeta::query()->where('user_id', $teacherId)
                ->where('name', 'signature')
                ->first();
            $instructorSignatureImg = (!empty($instructorSignature) and !empty($instructorSignature->value)) ? url($instructorSignature->value) : null;

            if (!empty($instructorSignatureImg)) {
                $instructorSignatureImg = "<img src='{$instructorSignatureImg}' style='max-width: 100%; max-height: 100%'/>";
            }
        }

        $body = str_replace('[instructor_signature]', $instructorSignatureImg, $body);

        $userCertificateAdditional = $user->userMetas->where('name', 'certificate_additional')->first();
        $userCertificateAdditionalValue = !empty($userCertificateAdditional) ? $userCertificateAdditional->value : null;
        $body = str_replace('[user_certificate_additional]', $userCertificateAdditionalValue, $body);

        return $body;
    }

    private function makeQrCode($template)
    {
        $size = 128;
        $elements = $template->elements;

        if (!empty($elements) and !empty($elements['qr_code']) and !empty($elements['qr_code']['image_size'])) {
            $size = (int)$elements['qr_code']['image_size'];
        }

        $url = url("/certificate_validation");
        return QrCode::size($size)->generate($url);
    }

    private function makeImage($certificateTemplate, $body)
    {
        $img = Image::make(public_path($certificateTemplate->image));


        if ($certificateTemplate->rtl) {
            $Arabic = new \I18N_Arabic('Glyphs');
            $body = $Arabic->utf8Glyphs($body);
        }

        $img->text($body, $certificateTemplate->position_x, $certificateTemplate->position_y, function ($font) use ($certificateTemplate) {
            $font->file($certificateTemplate->rtl ? public_path('assets/default/fonts/vazir/Vazir-Medium.ttf') : public_path('assets/default/fonts/Montserrat-Medium.ttf'));
            $font->size($certificateTemplate->font_size);
            $font->color($certificateTemplate->text_color);
            $font->align($certificateTemplate->rtl ? 'right' : 'left');
        });

        return $img;
    }

    public function makeCourseCertificate($certificate)
    {

        $template = CertificateTemplate::where('status', 'publish')
            ->where('type', 'course')
            ->first();

        $course = $certificate->webinar;

        if (!empty($template) and !empty($course)) {
            $user = $certificate->student;

            $userCertificate = $this->saveCourseCertificate($user, $course);
            $locale = app()->getLocale();
            $body = (!empty($template->translate($locale)) and !empty($template->translate($locale)->body)) ? $template->translate($locale)->body : $template->body;

            $body = $this->makeBody(
                $template,
                $userCertificate,
                $user,
                $body,
                $course->title,
                null,
                $course->teacher->id,
                $course->teacher->full_name,
                $course->duration);

            $data = [
                'body' => $body
            ];

            $html = (string)view()->make('admin.certificates.create_template.show_certificate', $data);
            return $this->generateCertificateImage($userCertificate, $html);
        }

        $toastData = [
            'title' => trans('public.request_failed'),
            'msg' => trans('update.no_certificate_template_is_defined_for_courses'),
            'status' => 'error'
        ];

        return redirect()->back()->with(['toast' => $toastData]);
    }

    public function makeBundleCertificate($certificate)
    {

        $template = CertificateTemplate::where('status', 'publish')
            ->where('type', 'bundle')
            ->first();

        $bundle = $certificate->bundle;

        if (!empty($template) and !empty($bundle)) {
            $user = $certificate->student;

            $userCertificate = $this->saveBundleCertificate($user, $bundle);
            $locale = app()->getLocale();
            $body = (!empty($template->translate($locale)) and !empty($template->translate($locale)->body)) ? $template->translate($locale)->body : $template->body;

            $body = $this->makeBody(
                $template,
                $userCertificate,
                $user,
                $body,
                $bundle->title,
                null,
                $bundle->teacher->id,
                $bundle->teacher->full_name,
                $bundle->duration);

            $data = [
                'body' => $body
            ];

            $html = (string)view()->make('admin.certificates.create_template.show_certificate', $data);
            return $this->generateCertificateImage($userCertificate, $html);
        }

        $toastData = [
            'title' => trans('public.request_failed'),
            'msg' => trans('update.no_certificate_template_is_defined_for_bundles'),
            'status' => 'error'
        ];

        return redirect()->back()->with(['toast' => $toastData]);
    }

    /**
     * Generate certificate using HTML2Canvas approach or GD library fallback
     * This method returns a view that will handle client-side certificate generation
     * and also automatically saves the certificate as an image for mobile apps
     */
    private function generateCertificateImage($certificate, $html)
    {
        try {
            // Always generate and save server-side image for mobile apps
            $this->generateAndSaveCertificateImage($certificate, $html);
            
            // Check if we should use server-side generation (GD library) or client-side (HTML2Canvas)
            $useServerSide = config('app.certificate_server_side', false);
            
            if ($useServerSide && extension_loaded('gd')) {
                return $this->generateCertificateWithGD($certificate, $html);
            }
            
            // Return the HTML2Canvas view for client-side generation
            $data = [
                'body' => $html,
                'certificate' => $certificate
            ];
            
            return view('admin.certificates.create_template.show_certificate_html2canvas', $data);
            
        } catch (\Exception $e) {
            \Log::error('Certificate generation failed: ' . $e->getMessage());
            
            $toastData = [
                'title' => trans('public.request_failed'),
                'msg' => trans('update.certificate_generation_failed'),
                'status' => 'error'
            ];
            
            return redirect()->back()->with(['toast' => $toastData]);
        }
    }
    
    /**
     * Generate and save certificate image automatically for mobile apps
     */
    private function generateAndSaveCertificateImage($certificate, $html)
    {
        try {
            // Generate image using GD library
            $imagePath = $this->generateCertificateImageFromHtml($html, $certificate->id);
            
            if ($imagePath && file_exists($imagePath)) {
                // Move to permanent storage
                $permanentPath = public_path("store/certificates/certificate_{$certificate->id}.png");
                $permanentDir = dirname($permanentPath);
                
                if (!file_exists($permanentDir)) {
                    mkdir($permanentDir, 0755, true);
                }
                
                // Copy file to permanent location
                copy($imagePath, $permanentPath);
                
                // Update certificate record with image path
                $certificate->update([
                    'image_path' => "store/certificates/certificate_{$certificate->id}.png"
                ]);
                
                // Clean up temporary file
                if (file_exists($imagePath)) {
                    unlink($imagePath);
                }
                
                \Log::info("Certificate image saved: certificate_{$certificate->id}.png");
            }
            
        } catch (\Exception $e) {
            \Log::error('Failed to save certificate image: ' . $e->getMessage());
        }
    }
    
    /**
     * Generate certificate image from HTML using GD library
     */
    private function generateCertificateImageFromHtml($html, $certificateId)
    {
        try {
            // Create image using GD library
            $width = 800;
            $height = 600;
            $image = imagecreatetruecolor($width, $height);
            
            // Set colors
            $white = imagecolorallocate($image, 255, 255, 255);
            $black = imagecolorallocate($image, 0, 0, 0);
            $blue = imagecolorallocate($image, 102, 126, 234);
            $purple = imagecolorallocate($image, 118, 75, 162);
            
            // Fill background with gradient
            $this->createGradientBackground($image, $width, $height, $blue, $purple);
            
            // Extract text content from HTML
            $textContent = strip_tags($html);
            
            // Add certificate content
            $this->addCertificateContent($image, $textContent, $certificateId);
            
            // Save image
            $outputPath = storage_path('app/temp/certificate_' . $certificateId . '.png');
            $tempDir = dirname($outputPath);
            
            if (!file_exists($tempDir)) {
                mkdir($tempDir, 0755, true);
            }
            
            imagepng($image, $outputPath);
            imagedestroy($image);
            
            return file_exists($outputPath) ? $outputPath : false;
            
        } catch (\Exception $e) {
            \Log::error('GD certificate generation failed: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Create gradient background
     */
    private function createGradientBackground($image, $width, $height, $color1, $color2)
    {
        for ($y = 0; $y < $height; $y++) {
            $ratio = $y / $height;
            $r = (int)((1 - $ratio) * 102 + $ratio * 118);
            $g = (int)((1 - $ratio) * 126 + $ratio * 75);
            $b = (int)((1 - $ratio) * 234 + $ratio * 162);
            
            $color = imagecolorallocate($image, $r, $g, $b);
            imageline($image, 0, $y, $width, $y, $color);
        }
    }
    
    /**
     * Add certificate content to image
     */
    private function addCertificateContent($image, $textContent, $certificateId)
    {
        $white = imagecolorallocate($image, 255, 255, 255);
        $black = imagecolorallocate($image, 0, 0, 0);
        $blue = imagecolorallocate($image, 102, 126, 234);
        
        // Create white background rectangle for content
        imagefilledrectangle($image, 50, 50, 750, 550, $white);
        
        // Add title
        $title = "Certificate of Completion";
        $font = public_path('assets/default/fonts/Montserrat-Medium.ttf');
        
        if (file_exists($font)) {
            imagettftext($image, 28, 0, 200, 120, $blue, $font, $title);
            
            // Add certificate ID
            $certId = "Certificate ID: " . $certificateId;
            imagettftext($image, 16, 0, 200, 160, $black, $font, $certId);
            
            // Add date
            $date = "Issued: " . date('F j, Y');
            imagettftext($image, 14, 0, 200, 190, $black, $font, $date);
            
            // Add main content (truncated for GD limitations)
            $lines = explode("\n", wordwrap($textContent, 50));
            $y = 250;
            foreach (array_slice($lines, 0, 8) as $line) {
                imagettftext($image, 12, 0, 100, $y, $black, $font, trim($line));
                $y += 30;
            }
        } else {
            // Fallback to built-in fonts
            imagestring($image, 5, 200, 120, $title, $blue);
            imagestring($image, 3, 200, 160, "Certificate ID: " . $certificateId, $black);
            imagestring($image, 3, 200, 190, "Issued: " . date('F j, Y'), $black);
        }
    }
    
    /**
     * Generate certificate using GD library (server-side, no exec() required)
     */
    private function generateCertificateWithGD($certificate, $html)
    {
        try {
            // Extract text content from HTML (simplified approach)
            $textContent = strip_tags($html);
            
            // Create image
            $width = 800;
            $height = 600;
            $image = imagecreatetruecolor($width, $height);
            
            // Set colors
            $white = imagecolorallocate($image, 255, 255, 255);
            $black = imagecolorallocate($image, 0, 0, 0);
            $gold = imagecolorallocate($image, 255, 215, 0);
            $blue = imagecolorallocate($image, 0, 123, 255);
            
            // Fill background
            imagefill($image, 0, 0, $white);
            
            // Add decorative border
            imagerectangle($image, 10, 10, $width-10, $height-10, $gold);
            imagerectangle($image, 15, 15, $width-15, $height-15, $black);
            
            // Add title
            $title = "Certificate of Completion";
            $font = public_path('assets/default/fonts/Montserrat-Medium.ttf');
            
            if (file_exists($font)) {
                imagettftext($image, 28, 0, 200, 80, $blue, $font, $title);
                
                // Add certificate ID
                $certId = "Certificate ID: " . $certificate->id;
                imagettftext($image, 16, 0, 200, 120, $black, $font, $certId);
                
                // Add date
                $date = "Issued: " . date('F j, Y', strtotime($certificate->created_at));
                imagettftext($image, 14, 0, 200, 150, $black, $font, $date);
                
                // Add main content (truncated for GD limitations)
                $lines = explode("\n", wordwrap($textContent, 50));
                $y = 200;
                foreach (array_slice($lines, 0, 10) as $line) {
                    imagettftext($image, 12, 0, 50, $y, $black, $font, trim($line));
                    $y += 25;
                }
            } else {
                // Fallback to built-in fonts
                imagestring($image, 5, 200, 80, $title, $blue);
                imagestring($image, 3, 200, 120, "Certificate ID: " . $certificate->id, $black);
                imagestring($image, 3, 200, 150, "Issued: " . date('F j, Y', strtotime($certificate->created_at)), $black);
            }
            
            // Save image
            $outputPath = storage_path('app/temp/certificate_' . $certificate->id . '.png');
            $tempDir = dirname($outputPath);
            
            if (!file_exists($tempDir)) {
                mkdir($tempDir, 0755, true);
            }
            
            imagepng($image, $outputPath);
            imagedestroy($image);
            
            if (file_exists($outputPath)) {
                $headers = array(
                    'Content-Type' => 'image/png',
                    'Content-Disposition' => 'attachment; filename="certificate.png"'
                );
                
                return response()->download($outputPath, "certificate.png", $headers)->deleteFileAfterSend();
            }
            
        } catch (\Exception $e) {
            \Log::error('GD certificate generation failed: ' . $e->getMessage());
        }
        
        // Fallback to HTML2Canvas if GD generation fails
        $data = [
            'body' => $html,
            'certificate' => $certificate
        ];
        
        return view('admin.certificates.create_template.show_certificate_html2canvas', $data);
    }
    


    // Legacy method - kept for backward compatibility but now redirects to new method
    private function sendToApi($certificate, $html)
    {
        return $this->generateCertificateImage($certificate, $html);
    }

    public function saveCourseCertificate($user, $course)
    {
        $certificate = Certificate::where('webinar_id', $course->id)
            ->where('student_id', $user->id)
            ->first();

        $data = [
            'webinar_id' => $course->id,
            'student_id' => $user->id,
            'type' => 'course',
            'created_at' => time()
        ];

        if (!empty($certificate)) {
            $certificate->update($data);
        } else {
            $certificate = Certificate::create($data);

            $notifyOptions = [
                '[c.title]' => $course->title,
            ];
            sendNotification('new_certificate', $notifyOptions, $user->id);
        }

        return $certificate;
    }

    public function saveBundleCertificate($user, $bundle)
    {
        $certificate = Certificate::where('bundle_id', $bundle->id)
            ->where('student_id', $user->id)
            ->first();

        $data = [
            'bundle_id' => $bundle->id,
            'student_id' => $user->id,
            'type' => 'bundle',
            'created_at' => time()
        ];

        if (!empty($certificate)) {
            $certificate->update($data);
        } else {
            $certificate = Certificate::create($data);

            $notifyOptions = [
                '[c.title]' => $bundle->title,
            ];
            sendNotification('new_certificate', $notifyOptions, $user->id);
        }

        return $certificate;
    }

}
