
import 'package:flutter/material.dart';
import 'package:webinar/app/models/certificate_model.dart';
import 'package:webinar/app/services/user_service/certificate_service.dart';
import 'package:webinar/common/common.dart';
import 'package:webinar/common/components.dart';
import 'package:webinar/common/utils/app_text.dart';
import 'package:webinar/common/utils/constants.dart';
import 'package:webinar/common/data/app_data.dart';
import 'package:webinar/common/utils/download_manager.dart';
import 'package:share_plus/share_plus.dart';
import 'package:path_provider/path_provider.dart';
import 'package:cached_network_image/cached_network_image.dart';
import 'dart:io';

class CertificatesDetailsPage extends StatefulWidget {
  static const String pageName = '/certificate-details';
  const CertificatesDetailsPage({super.key});

  @override
  State<CertificatesDetailsPage> createState() => _CertificatesDetailsPageState();
}

class _CertificatesDetailsPageState extends State<CertificatesDetailsPage> {

  CertificateModel? data;
  String? type;
  String? token;
  String? certificateImageUrl;
  bool isLoadingCertificate = false;

  @override
  void initState() {
    super.initState();
    
    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      data = (ModalRoute.of(context)!.settings.arguments as List)[0];
      type = (ModalRoute.of(context)!.settings.arguments as List)[1];
      print(type);

      if(type == 'achievements'){
        data?.link = data?.certificate?.link;
      }

      // Generate certificate image URL for mobile app
      _generateCertificateImageUrl();

      setState(() {});
    });

    AppData.getAccessToken().then((value){
      setState(() {
        token = value;
      });
    });
  }

  void _generateCertificateImageUrl() async {
    print('=== Certificate Data Debug ===');
    print('Data ID: ${data?.id}');
    print('Certificate ID: ${data?.certificate?.id}');
    print('User ID: ${data?.user?.id}');
    print('Type: $type');
    print('=============================');

    // Use the certificate ID to get the saved image URL
    int? certificateId = data?.certificate?.id ?? data?.id;
    int? userId = data?.user?.id;
    
    if (certificateId != null) {
      setState(() {
        isLoadingCertificate = true;
      });

      try {
        // First, try to get cached certificate
        String? cachedPath = await CertificateService.getCachedCertificatePath(certificateId);
        
        if (cachedPath != null && await File(cachedPath).exists()) {
          // Use cached certificate
          certificateImageUrl = cachedPath;
          print('Using cached certificate: $cachedPath');
        } else {
          // Try to download and cache the certificate
          String? downloadedPath = await CertificateService.downloadAndCacheCertificate(certificateId, token: token);
          
          if (downloadedPath != null) {
            certificateImageUrl = downloadedPath;
            print('Downloaded and cached certificate: $downloadedPath');
          } else if (userId != null) {
            // If download fails, try to generate the certificate
            print('Attempting to generate certificate from app...');
            String? generatedPath = await CertificateService.generateCertificateFromApp(
              certificateId: certificateId,
              userId: userId,
              type: type ?? 'quiz',
              token: token,
            );
            
            if (generatedPath != null) {
              certificateImageUrl = generatedPath;
              print('Generated certificate: $generatedPath');
            } else {
              // Fallback to online URL
              certificateImageUrl = CertificateService.getCertificateImageUrl(certificateId);
              print('Using online certificate URL: $certificateImageUrl');
            }
          } else {
            // Fallback to online URL
            certificateImageUrl = CertificateService.getCertificateImageUrl(certificateId);
            print('Using online certificate URL: $certificateImageUrl');
          }
        }
      } catch (e) {
        print('Error loading certificate: $e');
        // Fallback to online URL
        certificateImageUrl = CertificateService.getCertificateImageUrl(certificateId);
      } finally {
        setState(() {
          isLoadingCertificate = false;
        });
      }
    } else {
      print('No certificate ID found');
      certificateImageUrl = null;
    }
  }

  @override
  Widget build(BuildContext context) {
    return directionality(
      child: Scaffold(
        
        appBar: appbar(title: appText.certificateDetails),

        body: Stack(
          children: [

            Positioned.fill(
              child: SingleChildScrollView(
                physics: const BouncingScrollPhysics(),
                padding: padding(),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [

                    space(20),
              
                    Text(
                      type == 'completion'
                        ? data?.webinar?.title ?? ''
                        : data?.quiz?.title ?? '',
                      style: style16Bold(),
                    ),


                    if(type == 'achievements')...{
                      space(4),
                      
                      Text(
                        data?.webinar?.title ?? '',
                        style: style12Regular().copyWith(color: greyA5),
                      ),
                    },

                    space(20),

                    // Certificate Display
                    if(type == 'completion' || data?.certificate != null)...{
                      AspectRatio(
                        aspectRatio: 16 / 10,
                        child: ClipRRect(
                          borderRadius: borderRadius(),
                          child: _buildCertificateImage(),
                        ),
                      )
                    } else ...{
                      AspectRatio(
                        aspectRatio: 16 / 10,
                        child: ClipRRect(
                          borderRadius: borderRadius(),
                          child: fadeInImage(
                            data?.webinar?.image ?? '', 
                            getSize().width, 
                            getSize().width
                          ),
                        ),
                      )
                    },

                    space(30),


                    if(type == 'completion')...{
                      
                      Column(
                        children: [
                          
                          space(0,width: getSize().width),

                          Text(
                            appText.shareCertificate,
                            style: style20Bold(),
                          ),
                          
                          space(10),
                          
                          Text(
                            appText.shareCertificateDesc,
                            style: style14Regular().copyWith(color: greyA5),
                            textAlign: TextAlign.center,
                          ),

                        ],
                      )

                    },

                    space(30),

                    if(type == 'completion')...{
                      Row(
                        children: [

                          Expanded(
                            child: button(
                              onTap: () async {
                                if (certificateImageUrl != null) {
                                  downloadSheet(
                                    certificateImageUrl!, 
                                    'certificate_${data!.certificate?.id ?? data!.id}.png'
                                  );
                                } else {
                                  // Fallback to original link if certificate image URL is not available
                                  downloadSheet(
                                    data!.link!, 
                                    '${data!.date ?? data!.quiz?.title!.replaceAll(' ', '_')}.png'
                                  );
                                }
                              }, 
                              width: getSize().width, 
                              height: 51, 
                              text: appText.download, 
                              bgColor: green77(), 
                              textColor: Colors.white
                            ),
                          ),

                          space(0,width: 16),

                          Expanded(
                            child: button(
                              onTap: () async {
                                String directory = (await getApplicationSupportDirectory()).path;
                                String fileName = 'certificate_${data!.certificate?.id ?? data!.id}.png';

                                if(await DownloadManager.findFile(directory, fileName, isOpen: false)){
                                  Share.shareXFiles([XFile('$directory/$fileName')]);
                                }else{
                                  String? res;
                                  if (certificateImageUrl != null) {
                                    res = await downloadSheet(certificateImageUrl!, fileName, isOpen: false);
                                  } else {
                                    res = await downloadSheet(data!.link!, '${data!.date}.png', isOpen: false);
                                  }

                                  if (res != null) {
                                    Share.shareXFiles([XFile(res)]);
                                  }
                                }
                              }, 
                              width: getSize().width,
                              height: 52, 
                              text: appText.share, 
                              bgColor: Colors.white, 
                              textColor: green77(),
                              borderColor: green77()
                            ),
                          ),
                          
                        ],
                      ),
                    }

                  ],
                ),
              ),

            )

          ],
        ),

      )
    );
  }

  Widget _buildCertificateImage() {
    if (certificateImageUrl == null) {
      return Container(
        width: getSize().width,
        height: getSize().width,
        decoration: BoxDecoration(
          color: Colors.grey[200],
          borderRadius: borderRadius(),
        ),
        child: Center(
          child: Column(
            mainAxisAlignment: MainAxisAlignment.center,
            children: [
              Icon(Icons.image_not_supported, size: 50, color: Colors.grey[400]),
              space(10),
              Text(
                'Certificate not available',
                style: style14Regular().copyWith(color: Colors.grey[600]),
              ),
            ],
          ),
        ),
      );
    }

    return Container(
      width: getSize().width,
      height: getSize().width,
      decoration: BoxDecoration(
        color: Colors.white,
        borderRadius: borderRadius(),
        boxShadow: [
          BoxShadow(
            color: Colors.black.withOpacity(0.1),
            blurRadius: 10,
            offset: Offset(0, 5),
          ),
        ],
      ),
      child: ClipRRect(
        borderRadius: borderRadius(),
        child: _buildCertificateImageWidget(),
      ),
    );
  }

  Widget _buildCertificateImageWidget() {
    // Check if it's a local file path
    if (certificateImageUrl!.startsWith('/') || certificateImageUrl!.startsWith('file://')) {
      // Local file - use Image.file for highest quality
      return Image.file(
        File(certificateImageUrl!),
        fit: BoxFit.contain,
        width: getSize().width,
        height: getSize().width,
        errorBuilder: (context, error, stackTrace) {
          print('Local certificate image error: $error');
          return _buildErrorWidget();
        },
      );
    } else {
      // Network URL - use CachedNetworkImage for caching and quality
      return CachedNetworkImage(
        imageUrl: certificateImageUrl!,
        fit: BoxFit.contain,
        width: getSize().width,
        height: getSize().width,
        memCacheWidth: 1200, // High quality cache
        memCacheHeight: 900,
        placeholder: (context, url) => Container(
          width: getSize().width,
          height: getSize().width,
          decoration: BoxDecoration(
            color: Colors.grey[200],
            borderRadius: borderRadius(),
          ),
          child: Center(
            child: Column(
              mainAxisAlignment: MainAxisAlignment.center,
              children: [
                CircularProgressIndicator(),
                space(10),
                Text(
                  'Loading certificate...',
                  style: style12Regular().copyWith(color: Colors.grey[600]),
                ),
              ],
            ),
          ),
        ),
        errorWidget: (context, url, error) {
          print('Network certificate image error: $error');
          return _buildErrorWidget();
        },
      );
    }
  }

  Widget _buildErrorWidget() {
    return Container(
      width: getSize().width,
      height: getSize().width,
      decoration: BoxDecoration(
        color: Colors.grey[200],
        borderRadius: borderRadius(),
      ),
      child: Center(
        child: Column(
          mainAxisAlignment: MainAxisAlignment.center,
          children: [
            Icon(Icons.error_outline, size: 50, color: Colors.grey[400]),
            space(10),
            Text(
              'Failed to load certificate',
              style: style14Regular().copyWith(color: Colors.grey[600]),
            ),
            space(5),
            Text(
              'URL: ${certificateImageUrl?.substring(0, certificateImageUrl!.length > 50 ? 50 : certificateImageUrl!.length)}...',
              style: style10Regular().copyWith(color: Colors.grey[500]),
            ),
            space(5),
            TextButton(
              onPressed: () {
                setState(() {
                  _generateCertificateImageUrl();
                });
              },
              child: Text('Retry', style: style12Regular().copyWith(color: green77())),
            ),
          ],
        ),
      ),
    );
  }
}