import 'dart:convert';
import 'dart:io';

import 'package:http/http.dart' as http;
import 'package:path_provider/path_provider.dart';
import 'package:webinar/app/models/certificate_model.dart';

import '../../../common/enums/error_enum.dart';
import '../../../common/utils/constants.dart';
import '../../../common/utils/error_handler.dart';
import '../../../common/utils/http_handler.dart';

class CertificateService{

  static Future<List<CertificateModel>> getAchievements()async{
    List<CertificateModel> data = [];
    try{
      String url = '${Constants.baseUrl}panel/certificates/achievements';

      Response res = await httpGetWithToken(url);
      
      var jsonResponse = jsonDecode(res.body);
      
      if(jsonResponse['success']){
        jsonResponse['data'].forEach((json){
          data.add(CertificateModel.fromJson(json));
        });
        return data;
      }else{
        ErrorHandler().showError(ErrorEnum.error, jsonResponse);
        return data;
      }

    }catch(e){
      return data;
    }
  }

  static Future<List<CertificateModel>> getCompletion()async{
    List<CertificateModel> data = [];
    try{
      String url = '${Constants.baseUrl}panel/webinars/certificates';

      Response res = await httpGetWithToken(url);
      
      var jsonResponse = jsonDecode(res.body);
      
      if(jsonResponse['success']){
        jsonResponse['data']['certificates'].forEach((json){
          data.add(CertificateModel.fromJson(json));
        });
        return data;
      }else{
        ErrorHandler().showError(ErrorEnum.error, jsonResponse);
        return data;
      }

    }catch(e){
      return data;
    }
  }

  static Future<List<CertificateModel>> getClassData()async{
    List<CertificateModel> data = [];
    try{
      String url = '${Constants.baseUrl}panel/certificates/created';

      Response res = await httpGetWithToken(url);
      
      var jsonResponse = jsonDecode(res.body);
      
      if(jsonResponse['success']){
        jsonResponse['data']['certificates'].forEach((json){
          data.add(CertificateModel.fromJson(json));
        });
        return data;
      }else{
        ErrorHandler().showError(ErrorEnum.error, jsonResponse);
        return data;
      }

    }catch(e){
      return data;
    }
  }

  /// Get the direct URL to the certificate image (for use with Image.network or CachedNetworkImage)
  static String getCertificateImageUrl(int certificateId) {
    // This should point to the saved certificate image in public/store/certificates
    // The URL should be: https://neft.sa/store/certificates/certificate_[ID].png
    String url = 'https://neft.sa/store/certificates/certificate_$certificateId.png';
    print('Generated certificate URL: $url');
    return url;
  }

  /// Force generate certificate image if it doesn't exist
  static Future<String?> generateCertificateImageIfNeeded({
    required int certificateId,
    required int userId,
    String type = 'quiz',
  }) async {
    try {
      String url = '${Constants.baseUrl}development/certificates/generate-image?certificate_id=$certificateId&user_id=$userId&type=$type';
      
      final response = await httpGetWithToken(url);
      
      if (response.statusCode == 200) {
        // Return the URL to the generated image
        return getCertificateImageUrl(certificateId);
      }
      return null;
    } catch (e) {
      print('Error generating certificate image: $e');
      return null;
    }
  }

  /// Get certificate HTML content for WebView display
  static Future<Map<String, dynamic>?> getCertificateHtml({
    required int certificateId,
    required int userId,
    String type = 'quiz',
  }) async {
    try {
      String url = '${Constants.baseUrl}development/certificates/get-html?certificate_id=$certificateId&user_id=$userId&type=$type';
      
      final response = await httpGetWithToken(url);
      final jsonResponse = jsonDecode(response.body);
      
      if (jsonResponse['status'] == true) {
        return jsonResponse['data'];
      } else {
        print('Failed to get certificate HTML: ${jsonResponse['message']}');
        return null;
      }
    } catch (e) {
      print('Error getting certificate HTML: $e');
      return null;
    }
  }

  /// Get certificate HTML content with authentication
  static Future<Map<String, dynamic>?> getCertificateHtmlWithAuth({
    required int certificateId,
    required int userId,
    String type = 'quiz',
    String? token,
  }) async {
    try {
      String url = '${Constants.baseUrl}development/certificates/get-html?certificate_id=$certificateId&user_id=$userId&type=$type';
      
      // Prepare headers
      Map<String, String> headers = {
        'Accept': 'application/json',
        'Content-Type': 'application/json',
        'x-api-key': Constants.apiKey,
      };

      // Add authorization header if token is available
      if (token != null && token.isNotEmpty) {
        headers['Authorization'] = 'Bearer $token';
      }

      final response = await http.get(Uri.parse(url), headers: headers);
      final jsonResponse = jsonDecode(response.body);
      
      if (jsonResponse['status'] == true) {
        return jsonResponse['data'];
      } else {
        print('Failed to get certificate HTML: ${jsonResponse['message']}');
        return null;
      }
    } catch (e) {
      print('Error getting certificate HTML with auth: $e');
      return null;
    }
  }

  /// Download certificate image from the backend API
  /// Returns the local file path, or null on failure
  static Future<String?> downloadCertificateImage(int certificateId, int userId, {String type = 'quiz'}) async {
    try {
      String url = '${Constants.baseUrl}development/certificates/generate-image?certificate_id=$certificateId&user_id=$userId&type=$type';
      final response = await http.get(Uri.parse(url));
      
      if (response.statusCode == 200) {
        Directory appDir = await getApplicationDocumentsDirectory();
        Directory certificatesDir = Directory('${appDir.path}/certificates');
        if (!await certificatesDir.exists()) {
          await certificatesDir.create(recursive: true);
        }
        String imagePath = '${certificatesDir.path}/certificate_$certificateId.png';
        File imageFile = File(imagePath);
        await imageFile.writeAsBytes(response.bodyBytes);
        return imagePath;
      }
      return null;
    } catch (e) {
      print('Error downloading certificate image: $e');
      return null;
    }
  }

  /// Safely download certificate image with authentication and error handling
  /// Returns the local file path, or null on failure
  static Future<String?> downloadCertificateWithAuth(String imageUrl, String? token) async {
    try {
      // Check if we already have a cached version
      Directory appDir = await getApplicationDocumentsDirectory();
      Directory certificatesDir = Directory('${appDir.path}/certificates');
      if (!await certificatesDir.exists()) {
        await certificatesDir.create(recursive: true);
      }

      // Create a unique filename based on the URL
      String fileName = 'certificate_${DateTime.now().millisecondsSinceEpoch}.png';
      String imagePath = '${certificatesDir.path}/$fileName';
      File imageFile = File(imagePath);

      // Prepare headers
      Map<String, String> headers = {
        'Accept': 'image/*',
        'Content-Type': 'application/json',
        'x-api-key': Constants.apiKey,
      };

      // Add authorization header if token is available
      if (token != null && token.isNotEmpty) {
        headers['Authorization'] = 'Bearer $token';
      }

      // Download the image
      final response = await http.get(
        Uri.parse(imageUrl),
        headers: headers,
      );

      // Check if the response is successful and contains image data
      if (response.statusCode == 200 && response.bodyBytes.isNotEmpty) {
        await imageFile.writeAsBytes(response.bodyBytes);
        return imagePath;
      } else {
        print('Failed to download certificate image. Status: ${response.statusCode}');
        return null;
      }
    } catch (e) {
      print('Error downloading certificate with auth: $e');
      return null;
    }
  }

  /// Download and cache certificate image locally for offline access
  static Future<String?> downloadAndCacheCertificate(int certificateId, {String? token}) async {
    try {
      // Check if already cached
      String? cachedPath = await getCachedCertificatePath(certificateId);
      if (cachedPath != null && await File(cachedPath).exists()) {
        print('Certificate $certificateId already cached at: $cachedPath');
        return cachedPath;
      }

      // Download the certificate image
      String url = getCertificateImageUrl(certificateId);
      print('Downloading certificate from: $url');

      // Prepare headers for authentication
      Map<String, String> headers = {
        'Accept': 'image/*',
        'User-Agent': 'Mozilla/5.0 (iPhone; CPU iPhone OS 14_0 like Mac OS X) AppleWebKit/605.1.15',
      };

      if (token != null && token.isNotEmpty) {
        headers['Authorization'] = 'Bearer $token';
      }

      final response = await http.get(Uri.parse(url), headers: headers);

      if (response.statusCode == 200 && response.bodyBytes.isNotEmpty) {
        // Save to local cache
        String localPath = await saveCertificateToCache(certificateId, response.bodyBytes);
        print('Certificate $certificateId cached successfully at: $localPath');
        return localPath;
      } else {
        print('Failed to download certificate $certificateId. Status: ${response.statusCode}');
        return null;
      }
    } catch (e) {
      print('Error downloading certificate $certificateId: $e');
      return null;
    }
  }

  /// Get the local cached path for a certificate
  static Future<String?> getCachedCertificatePath(int certificateId) async {
    try {
      Directory appDir = await getApplicationDocumentsDirectory();
      Directory certificatesDir = Directory('${appDir.path}/certificates');
      
      if (await certificatesDir.exists()) {
        String filePath = '${certificatesDir.path}/certificate_$certificateId.png';
        if (await File(filePath).exists()) {
          return filePath;
        }
      }
      return null;
    } catch (e) {
      print('Error getting cached certificate path: $e');
      return null;
    }
  }

  /// Save certificate image to local cache
  static Future<String> saveCertificateToCache(int certificateId, List<int> imageBytes) async {
    Directory appDir = await getApplicationDocumentsDirectory();
    Directory certificatesDir = Directory('${appDir.path}/certificates');
    
    if (!await certificatesDir.exists()) {
      await certificatesDir.create(recursive: true);
    }

    String filePath = '${certificatesDir.path}/certificate_$certificateId.png';
    File file = File(filePath);
    await file.writeAsBytes(imageBytes);
    
    return filePath;
  }

  /// Clear all cached certificates
  static Future<void> clearCertificateCache() async {
    try {
      Directory appDir = await getApplicationDocumentsDirectory();
      Directory certificatesDir = Directory('${appDir.path}/certificates');
      
      if (await certificatesDir.exists()) {
        await certificatesDir.delete(recursive: true);
        print('Certificate cache cleared');
      }
    } catch (e) {
      print('Error clearing certificate cache: $e');
    }
  }

  /// Generate certificate image directly from the app (without needing to open on website first)
  static Future<String?> generateCertificateFromApp({
    required int certificateId,
    required int userId,
    String type = 'quiz',
    String? token,
  }) async {
    try {
      print('Generating certificate $certificateId for user $userId from app...');
      
      // Call the backend API to generate the certificate
      String url = 'https://neft.sa/api/development/certificates/generate-image?certificate_id=$certificateId&user_id=$userId&type=$type';
      
      Map<String, String> headers = {
        'Accept': 'image/*',
        'Content-Type': 'application/json',
        'User-Agent': 'Mozilla/5.0 (iPhone; CPU iPhone OS 14_0 like Mac OS X) AppleWebKit/605.1.15',
      };

      if (token != null && token.isNotEmpty) {
        headers['Authorization'] = 'Bearer $token';
      }

      final response = await http.get(Uri.parse(url), headers: headers);

      if (response.statusCode == 200 && response.bodyBytes.isNotEmpty) {
        print('Certificate $certificateId generated successfully');
        
        // Save to local cache for offline access
        String localPath = await saveCertificateToCache(certificateId, response.bodyBytes);
        return localPath;
      } else {
        print('Failed to generate certificate $certificateId. Status: ${response.statusCode}');
        return null;
      }
    } catch (e) {
      print('Error generating certificate $certificateId: $e');
      return null;
    }
  }
}