// Certificate Generator using HTML2Canvas
class CertificateGenerator {
    constructor() {
        this.html2canvas = null;
        this.loadHtml2Canvas();
    }

    async loadHtml2Canvas() {
        // Load HTML2Canvas from CDN if not already loaded
        if (typeof html2canvas === 'undefined') {
            const script = document.createElement('script');
            script.src = 'https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js';
            script.onload = () => {
                this.html2canvas = html2canvas;
            };
            document.head.appendChild(script);
        } else {
            this.html2canvas = html2canvas;
        }
    }

    async generateCertificate(htmlContent, options = {}) {
        // Wait for HTML2Canvas to be loaded
        while (!this.html2canvas) {
            await new Promise(resolve => setTimeout(resolve, 100));
        }

        // Create a temporary container
        const container = document.createElement('div');
        container.style.position = 'absolute';
        container.style.left = '-9999px';
        container.style.top = '-9999px';
        container.innerHTML = htmlContent;
        document.body.appendChild(container);

        try {
            // Configure HTML2Canvas options
            const canvasOptions = {
                allowTaint: true,
                useCORS: true,
                scale: 2, // Higher quality
                width: options.width || 800,
                height: options.height || 600,
                backgroundColor: '#ffffff',
                ...options
            };

            // Generate canvas
            const canvas = await this.html2canvas(container, canvasOptions);
            
            // Convert canvas to blob
            return new Promise((resolve, reject) => {
                canvas.toBlob((blob) => {
                    if (blob) {
                        resolve(blob);
                    } else {
                        reject(new Error('Failed to generate certificate blob'));
                    }
                }, 'image/png', 1.0);
            });

        } finally {
            // Clean up
            document.body.removeChild(container);
        }
    }

    async downloadCertificate(htmlContent, filename = 'certificate.png', options = {}) {
        try {
            const blob = await this.generateCertificate(htmlContent, options);
            
            // Create download link
            const url = URL.createObjectURL(blob);
            const link = document.createElement('a');
            link.href = url;
            link.download = filename;
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            
            // Clean up URL
            URL.revokeObjectURL(url);
            
            return true;
        } catch (error) {
            console.error('Certificate generation failed:', error);
            return false;
        }
    }

    async getCertificateAsDataURL(htmlContent, options = {}) {
        try {
            const blob = await this.generateCertificate(htmlContent, options);
            return URL.createObjectURL(blob);
        } catch (error) {
            console.error('Certificate generation failed:', error);
            return null;
        }
    }
}

// Global instance
window.certificateGenerator = new CertificateGenerator(); 