@php
    $rtlLanguages = !empty($generalSettings['rtl_languages']) ? $generalSettings['rtl_languages'] : [];
    $isRtl = ((in_array(mb_strtoupper(app()->getLocale()), $rtlLanguages)) or (!empty($generalSettings['rtl_layout']) and $generalSettings['rtl_layout'] == 1));

    $certificateLtrFont = getCertificateMainSettings('ltr_font');
    $certificateRtlFont = getCertificateMainSettings('rtl_font');
@endphp

<!doctype html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
    <meta name="viewport" content="width=device-width, user-scalable=no, initial-scale=1.0, maximum-scale=1.0, minimum-scale=1.0">
    <meta http-equiv="X-UA-Compatible" content="ie=edge">
    <title>Certificate Generator</title>

    <style>
        @if(!empty($certificateLtrFont))
            @font-face {
                font-family: 'ltr-font-name';
                font-style: normal;
                font-weight: 400;
                font-display: swap;
                src: url({{ url($certificateLtrFont) }});
            }
        @endif

        @if(!empty($certificateRtlFont))
            @font-face {
                font-family: 'rtl-font-name';
                font-style: normal;
                font-weight: 400;
                font-display: swap;
                src: url({{ url($certificateRtlFont) }});
            }
        @endif

        body {
            display: flex;
            align-items: center;
            justify-content: center;
            font-family: 'ltr-font-name';
            margin: 0;
            padding: 20px;
            background-color: #f5f5f5;
        }

        body.rtl {
            direction: rtl;
            font-family: 'rtl-font-name';
        }

        .certificate-container {
            text-align: center;
        }

        .certificate-template-container {
            width: {{ \App\Models\CertificateTemplate::$templateWidth }}px;
            height: {{ \App\Models\CertificateTemplate::$templateHeight }}px;
            position: relative;
            border: 2px solid #000;
            background-repeat: no-repeat;
            background-size: contain;
            margin: 0 auto 20px;
            background-color: #ffffff;
        }

        .certificate-template-container .draggable-element {
            position: absolute !important;
            display: inline-block;
            white-space: pre-wrap;
        }

        .controls {
            margin-top: 20px;
        }

        .btn {
            background-color: #007bff;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            margin: 0 10px;
            font-size: 16px;
        }

        .btn:hover {
            background-color: #0056b3;
        }

        .btn:disabled {
            background-color: #6c757d;
            cursor: not-allowed;
        }

        .loading {
            display: none;
            margin: 10px 0;
        }

        .preview-image {
            max-width: 100%;
            margin-top: 20px;
            border: 1px solid #ddd;
            border-radius: 5px;
        }
    </style>
</head>
<body class="{{ $isRtl ? 'rtl' : '' }}">
    <div class="certificate-container">
        <div id="certificate-content" class="certificate-template-container">
            {!! $body !!}
        </div>

        <!-- <div class="controls">
            <button id="generate-btn" class="btn">Generate Certificate</button>
            <button id="download-btn" class="btn" style="display: none;">Download Certificate</button>
            <button id="preview-btn" class="btn">Preview</button>
        </div> -->

        <div id="loading" class="loading">
            <p>Generating certificate...</p>
        </div>

        <div id="preview-container"></div>
    </div>

    <!-- Include HTML2Canvas and our certificate generator -->
    <script src="{{ asset('assets/default/js/certificate-generator.js') }}"></script>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const generateBtn = document.getElementById('generate-btn');
            const downloadBtn = document.getElementById('download-btn');
            const previewBtn = document.getElementById('preview-btn');
            const loading = document.getElementById('loading');
            const previewContainer = document.getElementById('preview-container');
            const certificateContent = document.getElementById('certificate-content');

            let generatedBlob = null;

            // Generate certificate
            generateBtn.addEventListener('click', async function() {
                generateBtn.disabled = true;
                loading.style.display = 'block';
                
                try {
                    const htmlContent = certificateContent.outerHTML;
                    const options = {
                        width: {{ \App\Models\CertificateTemplate::$templateWidth }},
                        height: {{ \App\Models\CertificateTemplate::$templateHeight }},
                        scale: 2
                    };

                    generatedBlob = await window.certificateGenerator.generateCertificate(htmlContent, options);
                    
                    downloadBtn.style.display = 'inline-block';
                    alert('Certificate generated successfully! Click Download to save it.');
                    
                } catch (error) {
                    console.error('Generation failed:', error);
                    alert('Failed to generate certificate. Please try again.');
                } finally {
                    generateBtn.disabled = false;
                    loading.style.display = 'none';
                }
            });

            // Download certificate
            downloadBtn.addEventListener('click', function() {
                if (generatedBlob) {
                    const url = URL.createObjectURL(generatedBlob);
                    const link = document.createElement('a');
                    link.href = url;
                    link.download = 'certificate.png';
                    document.body.appendChild(link);
                    link.click();
                    document.body.removeChild(link);
                    URL.revokeObjectURL(url);
                }
            });

            // Preview certificate
            previewBtn.addEventListener('click', async function() {
                previewBtn.disabled = true;
                loading.style.display = 'block';
                
                try {
                    const htmlContent = certificateContent.outerHTML;
                    const options = {
                        width: {{ \App\Models\CertificateTemplate::$templateWidth }},
                        height: {{ \App\Models\CertificateTemplate::$templateHeight }},
                        scale: 1
                    };

                    const dataUrl = await window.certificateGenerator.getCertificateAsDataURL(htmlContent, options);
                    
                    if (dataUrl) {
                        previewContainer.innerHTML = '<img src="' + dataUrl + '" class="preview-image" alt="Certificate Preview">';
                    }
                    
                } catch (error) {
                    console.error('Preview failed:', error);
                    alert('Failed to generate preview. Please try again.');
                } finally {
                    previewBtn.disabled = false;
                    loading.style.display = 'none';
                }
            });
        });
    </script>
</body>
</html> 