# Certificate Generation with HTML2Canvas

This implementation provides a solution for generating certificates without relying on server-side CLI tools or APIs that require `exec()` function, which is often blocked on shared hosting environments.

## Features

- **Client-side Generation**: Uses HTML2Canvas to generate certificates in the browser
- **Server-side Fallback**: GD library fallback for server-side generation (optional)
- **No External Dependencies**: No need for Puppeteer, wkhtmltoimage, or Chrome headless
- **High Quality**: Supports high-resolution certificate generation
- **Cross-browser Compatible**: Works in all modern browsers

## How It Works

### 1. HTML2Canvas Approach (Default)
- Certificate HTML is rendered in the browser
- HTML2Canvas converts the HTML to a canvas element
- Canvas is converted to a PNG image
- User can download the generated certificate
- Option to save to server via AJAX

### 2. GD Library Fallback (Optional)
- Server-side generation using PHP's GD library
- No external dependencies required
- Simpler certificate design due to GD limitations
- Can be enabled via configuration

## Configuration

### Environment Variables

Add to your `.env` file:

```env
# Set to true for server-side GD generation, false for client-side HTML2Canvas
CERTIFICATE_SERVER_SIDE=false
```

### Configuration File

The setting is also available in `config/app.php`:

```php
'certificate_server_side' => env('CERTIFICATE_SERVER_SIDE', false),
```

## Files Created/Modified

### New Files
1. `public/assets/default/js/certificate-generator.js` - HTML2Canvas wrapper class
2. `resources/views/admin/certificates/create_template/show_certificate_html2canvas.blade.php` - New certificate view with HTML2Canvas functionality
3. `CERTIFICATE_GENERATION_README.md` - This documentation file

### Modified Files
1. `app/Mixins/Certificate/MakeCertificate.php` - Updated certificate generation logic
2. `config/app.php` - Added certificate configuration option
3. `app/Http/Controllers/Panel/CertificateController.php` - Added test method
4. `routes/web.php` - Added test route

## Usage

### For Existing Certificate Generation

The existing certificate generation methods (`makeQuizCertificate`, `makeCourseCertificate`, `makeBundleCertificate`) now automatically use the new system:

```php
$makeCertificate = new MakeCertificate();
$result = $makeCertificate->makeQuizCertificate($quizResult);
```

### Testing the System

Visit the test route to see the HTML2Canvas certificate generation in action:

```
/test-certificate-generation
```

This route requires authentication and will show a sample certificate with generation controls.

### Custom Implementation

To use the HTML2Canvas system in your own code:

```php
// In your controller
public function generateCertificate()
{
    $data = [
        'body' => $certificateHtml,
        'certificate' => $certificateObject
    ];
    
    return view('admin.certificates.create_template.show_certificate_html2canvas', $data);
}
```

## JavaScript API

The certificate generator provides a JavaScript API:

```javascript
// Generate certificate
const blob = await window.certificateGenerator.generateCertificate(htmlContent, options);

// Download certificate
await window.certificateGenerator.downloadCertificate(htmlContent, 'certificate.png', options);

// Get as data URL
const dataUrl = await window.certificateGenerator.getCertificateAsDataURL(htmlContent, options);
```

### Options

```javascript
const options = {
    width: 800,           // Certificate width
    height: 600,          // Certificate height
    scale: 2,             // Quality multiplier (higher = better quality)
    backgroundColor: '#ffffff', // Background color
    allowTaint: true,     // Allow cross-origin images
    useCORS: true         // Use CORS for external resources
};
```

## Saving to Server

The system includes a method to save generated certificates to the server:

```javascript
// After generating certificate
const blob = await window.certificateGenerator.generateCertificate(htmlContent);
const formData = new FormData();
formData.append('image', blob);
formData.append('certificate_id', certificateId);

fetch('/certificate_validation/certificates/save-image', {
    method: 'POST',
    body: formData,
    headers: {
        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
    }
})
.then(response => response.json())
.then(data => {
    if (data.status) {
        console.log('Certificate saved:', data.url);
    }
});
```

## Browser Compatibility

- Chrome 60+
- Firefox 55+
- Safari 12+
- Edge 79+

## Performance Considerations

- **Client-side**: Processing happens in the user's browser, reducing server load
- **Memory**: Large certificates may use significant browser memory
- **Quality**: Higher scale values increase quality but also processing time
- **Network**: Generated images are typically 100KB-1MB depending on quality

## Troubleshooting

### Common Issues

1. **HTML2Canvas not loading**
   - Check if CDN is accessible
   - Verify JavaScript console for errors

2. **Fonts not rendering correctly**
   - Ensure fonts are loaded before generation
   - Use web-safe fonts as fallback

3. **Images not appearing**
   - Check CORS settings for external images
   - Ensure images are accessible

4. **Large file sizes**
   - Reduce scale value
   - Optimize certificate design

### Debug Mode

Enable browser developer tools to see detailed error messages and generation progress.

## Security Considerations

- Generated certificates are processed client-side
- No sensitive data is sent to external services
- CSRF protection is included for server saves
- File uploads are validated and sanitized

## Future Enhancements

- PDF generation support
- Batch certificate generation
- Custom certificate templates
- Watermark support
- Digital signature integration

## Mobile App Integration

For mobile app integration that requires actual image files (PNG/JPG) instead of HTML, see the separate [Mobile App Integration Guide](MOBILE_APP_INTEGRATION.md).

### Quick Mobile API Reference:

- **Generate Certificate Image**: `GET /api/development/certificates/generate-image`
- **Get Certificate List**: `GET /api/development/certificates/list`

The mobile API returns actual PNG image files that can be displayed directly in mobile apps without requiring web views or HTML rendering.

## Support

For issues or questions about the certificate generation system, please refer to the HTML2Canvas documentation or create an issue in the project repository. 