# Mobile App Certificate Integration Guide

This guide explains how to integrate certificate generation in your mobile app using the new API endpoints that return actual image files and HTML content for WebView display.

## API Endpoints

### 1. Get Certificate Image
**GET** `/api/development/certificates/generate-image`

Returns a PNG image file of the certificate. If the image doesn't exist, it will be generated automatically and saved for future use.

#### Parameters:
- `certificate_id` (required): The certificate ID
- `user_id` (required): The user ID
- `type` (optional): Certificate type - `quiz`, `course`, or `bundle` (default: `quiz`)

#### Response:
Returns a PNG image file with headers:
```
Content-Type: image/png
Content-Disposition: inline; filename="certificate.png"
```

#### Example Request:
```
GET /api/development/certificates/generate-image?certificate_id=123&user_id=456&type=quiz
```

### 2. Get Certificate HTML (NEW)
**GET** `/api/development/certificates/get-html`

Returns HTML content optimized for mobile WebView display. This provides the same beautiful certificate design as the website using HTML2Canvas.

#### Parameters:
- `certificate_id` (required): The certificate ID
- `user_id` (required): The user ID
- `type` (optional): Certificate type - `quiz`, `course`, or `bundle` (default: `quiz`)

#### Response:
```json
{
    "status": true,
    "data": {
        "html": "<!DOCTYPE html>...",
        "certificate_id": 123,
        "type": "quiz"
    }
}
```

#### Example Request:
```
GET /api/development/certificates/get-html?certificate_id=123&user_id=456&type=quiz
```

### 3. Get Certificate List
**GET** `/api/development/certificates/list`

Returns a list of user's certificates.

#### Parameters:
- `user_id` (required): The user ID

#### Response:
```json
{
    "status": true,
    "data": [
        {
            "id": 123,
            "type": "quiz",
            "title": "Course Title",
            "created_at": "2024-01-15T10:30:00.000000Z",
            "image_url": "https://example.com/store/certificates/certificate_123.png"
        }
    ]
}
```

## Mobile App Implementation Examples

### Flutter/Dart Example

```dart
import 'dart:io';
import 'package:http/http.dart' as http;
import 'package:flutter_inappwebview/flutter_inappwebview.dart';

class CertificateService {
  static const String baseUrl = 'https://your-domain.com/api/development';

  // Get certificate HTML for WebView display (RECOMMENDED)
  static Future<Map<String, dynamic>?> getCertificateHtml({
    required int certificateId,
    required int userId,
    String type = 'quiz',
    String? token,
  }) async {
    try {
      final url = Uri.parse(
        '$baseUrl/certificates/get-html?certificate_id=$certificateId&user_id=$userId&type=$type'
      );
      
      Map<String, String> headers = {
        'Accept': 'application/json',
        'Content-Type': 'application/json',
        'x-api-key': 'your-api-key',
      };

      if (token != null && token.isNotEmpty) {
        headers['Authorization'] = 'Bearer $token';
      }

      final response = await http.get(url, headers: headers);
      
      if (response.statusCode == 200) {
        final data = json.decode(response.body);
        if (data['status'] == true) {
          return data['data'];
        }
      }
    } catch (e) {
      print('Error getting certificate HTML: $e');
    }
    return null;
  }

  // Generate certificate image (FALLBACK)
  static Future<File?> generateCertificateImage({
    required int certificateId,
    required int userId,
    String type = 'quiz',
  }) async {
    try {
      final url = Uri.parse(
        '$baseUrl/certificates/generate-image?certificate_id=$certificateId&user_id=$userId&type=$type'
      );
      
      final response = await http.get(url);
      
      if (response.statusCode == 200) {
        // Save image to temporary file
        final tempDir = await getTemporaryDirectory();
        final file = File('${tempDir.path}/certificate_$certificateId.png');
        await file.writeAsBytes(response.bodyBytes);
        return file;
      }
    } catch (e) {
      print('Error generating certificate: $e');
    }
    return null;
  }

  // Get certificate list
  static Future<List<Certificate>> getCertificates(int userId) async {
    try {
      final url = Uri.parse('$baseUrl/certificates/list?user_id=$userId');
      final response = await http.get(url);
      
      if (response.statusCode == 200) {
        final data = json.decode(response.body);
        if (data['status']) {
          return (data['data'] as List)
              .map((item) => Certificate.fromJson(item))
              .toList();
        }
      }
    } catch (e) {
      print('Error fetching certificates: $e');
    }
    return [];
  }
}

// Certificate display widget using WebView
class CertificateDisplayWidget extends StatefulWidget {
  final int certificateId;
  final int userId;
  final String type;
  final String? token;

  const CertificateDisplayWidget({
    Key? key,
    required this.certificateId,
    required this.userId,
    this.type = 'quiz',
    this.token,
  }) : super(key: key);

  @override
  _CertificateDisplayWidgetState createState() => _CertificateDisplayWidgetState();
}

class _CertificateDisplayWidgetState extends State<CertificateDisplayWidget> {
  String? certificateHtml;
  bool isLoading = true;
  bool useWebView = true;

  @override
  void initState() {
    super.initState();
    _loadCertificateHtml();
  }

  Future<void> _loadCertificateHtml() async {
    try {
      final htmlData = await CertificateService.getCertificateHtml(
        certificateId: widget.certificateId,
        userId: widget.userId,
        type: widget.type,
        token: widget.token,
      );

      if (htmlData != null && htmlData['html'] != null) {
        setState(() {
          certificateHtml = htmlData['html'];
          isLoading = false;
        });
      } else {
        setState(() {
          isLoading = false;
        });
      }
    } catch (e) {
      setState(() {
        isLoading = false;
      });
    }
  }

  @override
  Widget build(BuildContext context) {
    if (isLoading) {
      return Center(child: CircularProgressIndicator());
    }

    if (useWebView && certificateHtml != null) {
      return Container(
        height: 400,
        child: InAppWebView(
          initialData: InAppWebViewInitialData(
            data: certificateHtml!,
            mimeType: 'text/html',
            encoding: 'UTF-8',
          ),
          initialSettings: InAppWebViewSettings(
            javaScriptEnabled: true,
            supportZoom: false,
            builtInZoomControls: false,
            displayZoomControls: false,
          ),
        ),
      );
    } else {
      // Fallback to image display
      return FutureBuilder<File?>(
        future: CertificateService.generateCertificateImage(
          certificateId: widget.certificateId,
          userId: widget.userId,
          type: widget.type,
        ),
        builder: (context, snapshot) {
          if (snapshot.connectionState == ConnectionState.waiting) {
            return Center(child: CircularProgressIndicator());
          }
          
          if (snapshot.hasData && snapshot.data != null) {
            return Image.file(snapshot.data!);
          }
          
          return Center(child: Text('Failed to load certificate'));
        },
      );
    }
  }
}
```

### React Native Example

```javascript
import { Platform } from 'react-native';
import RNFS from 'react-native-fs';

class CertificateService {
  static baseUrl = 'https://your-domain.com/api/development';

  // Generate certificate image
  static async generateCertificateImage(certificateId, userId, type = 'quiz') {
    try {
      const url = `${this.baseUrl}/certificates/generate-image?certificate_id=${certificateId}&user_id=${userId}&type=${type}`;
      
      const response = await fetch(url);
      
      if (response.ok) {
        const blob = await response.blob();
        const base64 = await this.blobToBase64(blob);
        
        // Save to device
        const fileName = `certificate_${certificateId}.png`;
        const filePath = `${RNFS.DocumentDirectoryPath}/${fileName}`;
        
        await RNFS.writeFile(filePath, base64, 'base64');
        return filePath;
      }
    } catch (error) {
      console.error('Error generating certificate:', error);
    }
    return null;
  }

  // Get certificate list
  static async getCertificates(userId) {
    try {
      const url = `${this.baseUrl}/certificates/list?user_id=${userId}`;
      const response = await fetch(url);
      
      if (response.ok) {
        const data = await response.json();
        if (data.status) {
          return data.data;
        }
      }
    } catch (error) {
      console.error('Error fetching certificates:', error);
    }
    return [];
  }

  // Helper: Convert blob to base64
  static blobToBase64(blob) {
    return new Promise((resolve, reject) => {
      const reader = new FileReader();
      reader.onload = () => resolve(reader.result.split(',')[1]);
      reader.onerror = reject;
      reader.readAsDataURL(blob);
    });
  }
}

export default CertificateService;
```

### iOS Swift Example

```swift
import Foundation
import UIKit

class CertificateService {
    static let baseUrl = "https://your-domain.com/api/development"
    
    // Generate certificate image
    static func generateCertificateImage(
        certificateId: Int,
        userId: Int,
        type: String = "quiz",
        completion: @escaping (UIImage?) -> Void
    ) {
        let urlString = "\(baseUrl)/certificates/generate-image?certificate_id=\(certificateId)&user_id=\(userId)&type=\(type)"
        
        guard let url = URL(string: urlString) else {
            completion(nil)
            return
        }
        
        URLSession.shared.dataTask(with: url) { data, response, error in
            guard let data = data, error == nil else {
                DispatchQueue.main.async {
                    completion(nil)
                }
                return
            }
            
            if let image = UIImage(data: data) {
                DispatchQueue.main.async {
                    completion(image)
                }
            } else {
                DispatchQueue.main.async {
                    completion(nil)
                }
            }
        }.resume()
    }
    
    // Get certificate list
    static func getCertificates(userId: Int, completion: @escaping ([Certificate]) -> Void) {
        let urlString = "\(baseUrl)/certificates/list?user_id=\(userId)"
        
        guard let url = URL(string: urlString) else {
            completion([])
            return
        }
        
        URLSession.shared.dataTask(with: url) { data, response, error in
            guard let data = data, error == nil else {
                DispatchQueue.main.async {
                    completion([])
                }
                return
            }
            
            do {
                if let json = try JSONSerialization.jsonObject(with: data) as? [String: Any],
                   let status = json["status"] as? Bool,
                   status,
                   let certificatesData = json["data"] as? [[String: Any]] {
                    
                    let certificates = certificatesData.compactMap { Certificate(from: $0) }
                    DispatchQueue.main.async {
                        completion(certificates)
                    }
                } else {
                    DispatchQueue.main.async {
                        completion([])
                    }
                }
            } catch {
                DispatchQueue.main.async {
                    completion([])
                }
            }
        }.resume()
    }
}

// Certificate model
struct Certificate {
    let id: Int
    let type: String
    let title: String
    let createdAt: Date
    let imageUrl: String?
    
    init?(from json: [String: Any]) {
        guard let id = json["id"] as? Int,
              let type = json["type"] as? String,
              let title = json["title"] as? String,
              let createdAtString = json["created_at"] as? String else {
            return nil
        }
        
        self.id = id
        self.type = type
        self.title = title
        self.imageUrl = json["image_url"] as? String
        
        let formatter = ISO8601DateFormatter()
        self.createdAt = formatter.date(from: createdAtString) ?? Date()
    }
}
```

### Android Kotlin Example

```kotlin
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import java.net.URL
import java.text.SimpleDateFormat
import java.util.*

class CertificateService {
    companion object {
        private const val BASE_URL = "https://your-domain.com/api/development"
    }

    // Generate certificate image
    suspend fun generateCertificateImage(
        certificateId: Int,
        userId: Int,
        type: String = "quiz"
    ): Bitmap? = withContext(Dispatchers.IO) {
        try {
            val urlString = "$BASE_URL/certificates/generate-image?certificate_id=$certificateId&user_id=$userId&type=$type"
            val url = URL(urlString)
            val connection = url.openConnection()
            
            val inputStream = connection.getInputStream()
            return@withContext BitmapFactory.decodeStream(inputStream)
        } catch (e: Exception) {
            e.printStackTrace()
            return@withContext null
        }
    }

    // Get certificate list
    suspend fun getCertificates(userId: Int): List<Certificate> = withContext(Dispatchers.IO) {
        try {
            val urlString = "$BASE_URL/certificates/list?user_id=$userId"
            val url = URL(urlString)
            val connection = url.openConnection()
            
            val inputStream = connection.getInputStream()
            val response = inputStream.bufferedReader().use { it.readText() }
            
            // Parse JSON response
            val jsonObject = JSONObject(response)
            if (jsonObject.getBoolean("status")) {
                val certificatesArray = jsonObject.getJSONArray("data")
                val certificates = mutableListOf<Certificate>()
                
                for (i in 0 until certificatesArray.length()) {
                    val certJson = certificatesArray.getJSONObject(i)
                    certificates.add(Certificate.fromJson(certJson))
                }
                
                return@withContext certificates
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return@withContext emptyList()
    }
}

// Certificate data class
data class Certificate(
    val id: Int,
    val type: String,
    val title: String,
    val createdAt: Date,
    val imageUrl: String?
) {
    companion object {
        fun fromJson(json: JSONObject): Certificate {
            val dateFormat = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'", Locale.getDefault())
            
            return Certificate(
                id = json.getInt("id"),
                type = json.getString("type"),
                title = json.getString("title"),
                createdAt = dateFormat.parse(json.getString("created_at")) ?: Date(),
                imageUrl = if (json.has("image_url")) json.getString("image_url") else null
            )
        }
    }
}
```

## Usage in Mobile App UI

### Display Certificate Image

```dart
// Flutter example
class CertificateScreen extends StatefulWidget {
  @override
  _CertificateScreenState createState() => _CertificateScreenState();
}

class _CertificateScreenState extends State<CertificateScreen> {
  File? certificateImage;
  bool isLoading = false;

  @override
  void initState() {
    super.initState();
    _loadCertificate();
  }

  Future<void> _loadCertificate() async {
    setState(() => isLoading = true);
    
    final image = await CertificateService.generateCertificateImage(
      certificateId: 123,
      userId: 456,
    );
    
    setState(() {
      certificateImage = image;
      isLoading = false;
    });
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(title: Text('Certificate')),
      body: Center(
        child: isLoading
            ? CircularProgressIndicator()
            : certificateImage != null
                ? Image.file(certificateImage!)
                : Text('Failed to load certificate'),
      ),
    );
  }
}
```

## Error Handling

### Common Error Responses

```json
// 400 Bad Request
{
    "status": false,
    "message": "Certificate ID and User ID are required"
}

// 404 Not Found
{
    "status": false,
    "message": "Certificate not found"
}

// 500 Internal Server Error
{
    "status": false,
    "message": "Certificate generation failed"
}
```

### Error Handling Example

```dart
Future<void> _loadCertificate() async {
  try {
    setState(() => isLoading = true);
    
    final image = await CertificateService.generateCertificateImage(
      certificateId: certificateId,
      userId: userId,
    );
    
    if (image != null) {
      setState(() {
        certificateImage = image;
        isLoading = false;
      });
    } else {
      setState(() {
        isLoading = false;
        errorMessage = 'Failed to generate certificate';
      });
    }
  } catch (e) {
    setState(() {
      isLoading = false;
      errorMessage = 'Error: ${e.toString()}';
    });
  }
}
```

## Best Practices

1. **Caching**: Cache generated certificates locally to avoid repeated API calls
2. **Loading States**: Show loading indicators while generating certificates
3. **Error Handling**: Implement proper error handling for network issues
4. **Image Quality**: Consider device screen density for optimal image display
5. **Offline Support**: Store certificates locally for offline viewing

## Testing

Test the API endpoints using tools like Postman or curl:

```bash
# Generate certificate image
curl -X GET "https://your-domain.com/api/development/certificates/generate-image?certificate_id=123&user_id=456&type=quiz" \
  -H "Accept: image/png" \
  --output certificate.png

# Get certificate list
curl -X GET "https://your-domain.com/api/development/certificates/list?user_id=456" \
  -H "Accept: application/json"
```

This integration provides a complete solution for displaying certificates as images in mobile apps without relying on web views or external dependencies. 

## ✅ **Complete Solution: HTML to Image Conversion & Mobile App Support**

### 🎯 **What We've Implemented:**

1. **Automatic Image Generation**: Every time a certificate is generated (web or mobile), it automatically creates and saves a PNG image
2. **Permanent Storage**: Images are saved in `/public/store/certificates/certificate_[ID].png`
3. **Mobile App Ready**: Mobile apps can directly access saved images without generation delays
4. **No `exec()` Function**: Uses PHP GD library instead of CLI tools

### 📁 **File Locations:**

| Purpose | Location | Cleanup |
|---------|----------|---------|
| **Temporary Generation** | `/storage/app/temp/` | Auto (1 day) |
| **Permanent Storage** | `/public/store/certificates/` | Manual |
| **Mobile App Access** | Direct URL: `https://your-domain.com/store/certificates/certificate_123.png` | Never |

### 🚀 **How It Works:**

#### **For New Certificates:**
1. User generates certificate (web or mobile)
2. System creates HTML certificate
3. **Automatically converts HTML to PNG image**
4. **Saves image permanently** in `/public/store/certificates/`
5. Updates database with image path
6. Returns image for immediate use

#### **For Mobile Apps:**
1. Mobile app requests certificate image
2. System checks if image exists
3. **If exists**: Returns saved image instantly
4. **If not exists**: Generates new image, saves it, then returns it
5. Future requests are instant

### 🔧 **Commands Available:**

#### **Generate Images for Existing Certificates:**
```bash
# Generate for all types (default)
php artisan certificates:generate-images

# Generate for specific type
php artisan certificates:generate-images --type=quiz

# Limit number of certificates
php artisan certificates:generate-images --limit=50

# Generate for specific type with limit
php artisan certificates:generate-images --type=course --limit=25
```

#### **Cleanup Temporary Files:**
```bash
# Clean files older than 1 day (default)
php artisan certificates:cleanup

# Clean files older than 7 days
php artisan certificates:cleanup --days=7
```

### 📱 **Mobile App Benefits:**

✅ **Instant Loading**: Saved images load immediately  
✅ **No Generation Delays**: No waiting for image creation  
✅ **Reliable Access**: Images are always available  
✅ **Better Performance**: Reduced server load  
✅ **Offline Support**: Can cache images locally  

### 📊 **Storage Management:**

```bash
# Check storage usage
du -sh /home/neftenergies/public_html/docroot.neft.sa/public/store/certificates/

# Count certificate images
find /home/neftenergies/public_html/docroot.neft.sa/public/store/certificates/ -name "*.png" | wc -l

# List all certificate images
ls -la /home/neftenergies/public_html/docroot.neft.sa/public/store/certificates/
```

### 🎯 **Result:**

Now your system:
- ✅ **Converts HTML certificates to images automatically**
- ✅ **Saves images permanently for mobile apps**
- ✅ **Works without `exec()` function**
- ✅ **Provides instant access for mobile apps**
- ✅ **Maintains web app functionality**
- ✅ **Handles existing certificates with batch generation**

Your mobile app developers can now access certificate images directly via:
```
https://neft.sa/store/certificates/certificate_123.png
```

Or through the API which will generate and save images automatically! 