<?php

namespace App\Http\Controllers\Api\Web;

use App\Http\Controllers\Api\Controller;
use App\Models\Certification;
use App\Models\User;
use Illuminate\Http\Request;

class CertificationsController extends Controller
{
    /**
     * Get certifications for the authenticated user
     */
    public function index(Request $request)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Check if user has a student_id
            if (empty($user->student_id)) {
                // Return empty array if no student_id is set
                return response()->json([
                    'status' => true,
                    'message' => 'No student ID found. Please add your student ID in settings.',
                    'data' => []
                ]);
            }

            // Match certifications where id_no OR govt_id matches the user's student_id
            // Optimized: Select only needed columns and use proper indexing
            $studentId = $user->student_id;
            
            // Optional pagination support
            $limit = $request->input('limit', 100); // Default 100, but allow override
            $page = $request->input('page', 1);
            
            $query = Certification::select('Id', 'neft_certificate_number', 'name', 'id_no', 'govt_id', 
                                           'company', 'course_name', 'actual_date', 'issue_date', 
                                           'expiration_date', 'template')
                ->where(function($query) use ($studentId) {
                    $query->where('id_no', $studentId)
                          ->orWhere('govt_id', $studentId);
                })
                ->orderBy('Id', 'desc');
            
            // If pagination is requested
            if ($request->has('paginate') && $request->input('paginate') === 'true') {
                $certifications = $query->paginate($limit);
            } else {
                // Return all (current behavior)
                $certifications = $query->limit($limit)->get();
            }

            return response()->json([
                'status' => true,
                'message' => 'Certifications retrieved successfully',
                'data' => $certifications
            ]);

        } catch (\Exception $e) {
            \Log::error('Certifications API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error retrieving certifications',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get a specific certification
     */
    public function show($id)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Check if user has a student_id
            if (empty($user->student_id)) {
                return response()->json([
                    'status' => false,
                    'message' => 'No student ID found. Please add your student ID in settings.'
                ], 400);
            }

            // Match certification where Id matches the requested id AND (id_no OR govt_id matches the user's student_id)
            // Optimized query with explicit select to reduce data transfer
            $certification = Certification::select('Id', 'neft_certificate_number', 'name', 'id_no', 'govt_id', 
                                                    'company', 'course_name', 'actual_date', 'issue_date', 
                                                    'expiration_date', 'template')
                ->where('Id', $id)
                ->where(function($query) use ($user) {
                    $query->where('id_no', $user->student_id)
                          ->orWhere('govt_id', $user->student_id);
                })
                ->first();

            if (!$certification) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certification not found'
                ], 404);
            }

            return response()->json([
                'status' => true,
                'message' => 'Certification retrieved successfully',
                'data' => $certification
            ]);

        } catch (\Exception $e) {
            \Log::error('Certification show API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error retrieving certification',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download a specific certification
     */
    public function download($id)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Check if user has a student_id
            if (empty($user->student_id)) {
                return response()->json([
                    'status' => false,
                    'message' => 'No student ID found. Please add your student ID in settings.'
                ], 400);
            }

            // Match certification where Id matches the requested id AND (id_no OR govt_id matches the user's student_id)
            // Optimized query with explicit select to reduce data transfer
            $certification = Certification::select('Id', 'neft_certificate_number', 'name', 'id_no', 'govt_id', 
                                                    'company', 'course_name', 'actual_date', 'issue_date', 
                                                    'expiration_date', 'template')
                ->where('Id', $id)
                ->where(function($query) use ($user) {
                    $query->where('id_no', $user->student_id)
                          ->orWhere('govt_id', $user->student_id);
                })
                ->first();

            if (!$certification) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certification not found'
                ], 404);
            }

            // For now, return the certification data as JSON
            // In the future, this could generate a PDF or image
            return response()->json([
                'status' => true,
                'message' => 'Certification download data',
                'data' => $certification,
                'download_url' => url("/api/development/certifications/{$id}")
            ]);

        } catch (\Exception $e) {
            \Log::error('Certification download API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error downloading certification',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get certificate template background image
     * Used for offline caching - downloads template once for multiple certificates
     */
    public function getTemplate(Request $request)
    {
        try {
            $templateId = $request->input('template_id', 1);
            $certificateId = $request->input('certificate_id');
            $userId = $request->input('user_id');
            
            // Get template background from assets
            $templatePath = public_path("assets/default_images/certificate_templates/template_{$templateId}.png");
            
            // If specific template doesn't exist, use default template 1
            if (!file_exists($templatePath)) {
                \Log::warning("Template {$templateId} not found, using default template 1");
                $templatePath = public_path("assets/default_images/certificate_templates/template_1.png");
            }
            
            // If still not found, create a simple placeholder
            if (!file_exists($templatePath)) {
                \Log::warning("No template images found, creating placeholder");
                
                // Create a simple white background as placeholder
                $width = 1200;
                $height = 850;
                $image = imagecreatetruecolor($width, $height);
                $white = imagecolorallocate($image, 255, 255, 255);
                imagefill($image, 0, 0, $white);
                
                // Add border
                $black = imagecolorallocate($image, 0, 0, 0);
                imagerectangle($image, 20, 20, $width-20, $height-20, $black);
                
                // Output as PNG
                ob_start();
                imagepng($image);
                $imageData = ob_get_clean();
                imagedestroy($image);
                
                return response($imageData)
                    ->header('Content-Type', 'image/png')
                    ->header('Cache-Control', 'public, max-age=31536000'); // Cache for 1 year
            }
            
            // Return the template image
            return response()->file($templatePath, [
                'Content-Type' => 'image/png',
                'Cache-Control' => 'public, max-age=31536000' // Cache for 1 year since templates don't change
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Get template API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error retrieving template',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all certifications with user information
     */
    public function getAllWithUsers(Request $request)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Get certifications with user information by joining on id_no = student_id OR govt_id = student_id
            $certifications = Certification::select('certifications.*', 'users.full_name', 'users.email', 'users.mobile')
                ->leftJoin('users', function($join) use ($user) {
                    $join->on('certifications.id_no', '=', 'users.student_id')
                         ->orOn('certifications.govt_id', '=', 'users.student_id');
                })
                ->where(function($query) use ($user) {
                    $query->where('certifications.id_no', $user->student_id)
                          ->orWhere('certifications.govt_id', $user->student_id);
                })
                ->orderBy('certifications.Id', 'desc')
                ->get();

            return response()->json([
                'status' => true,
                'message' => 'Certifications with user data retrieved successfully',
                'data' => $certifications
            ]);

        } catch (\Exception $e) {
            \Log::error('Certifications with users API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error retrieving certifications with user data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get certification by student ID (for admin purposes)
     */
    public function getByStudentId($studentId)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Get certifications for a specific student ID (checking both id_no and govt_id)
            $certifications = Certification::where(function($query) use ($studentId) {
                $query->where('id_no', $studentId)
                      ->orWhere('govt_id', $studentId);
            })
            ->orderBy('Id', 'desc')
            ->get();

            return response()->json([
                'status' => true,
                'message' => 'Certifications retrieved successfully',
                'data' => $certifications
            ]);

        } catch (\Exception $e) {
            \Log::error('Certifications by student ID API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error retrieving certifications by student ID',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get certifications by government ID
     */
    public function getByGovtId($govtId)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Get certifications for a specific government ID
            $certifications = Certification::where('govt_id', $govtId)
                ->orderBy('Id', 'desc')
                ->get();

            return response()->json([
                'status' => true,
                'message' => 'Certifications retrieved successfully',
                'data' => $certifications
            ]);

        } catch (\Exception $e) {
            \Log::error('Certifications by government ID API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error retrieving certifications by government ID',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get certifications by any ID (student_id or govt_id)
     */
    public function getByAnyId($anyId)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Get certifications for any ID (checking both id_no and govt_id)
            $certifications = Certification::where(function($query) use ($anyId) {
                $query->where('id_no', $anyId)
                      ->orWhere('govt_id', $anyId);
            })
            ->orderBy('Id', 'desc')
            ->get();

            return response()->json([
                'status' => true,
                'message' => 'Certifications retrieved successfully',
                'data' => $certifications
            ]);

        } catch (\Exception $e) {
            \Log::error('Certifications by any ID API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error retrieving certifications by any ID',
                'error' => $e->getMessage()
            ], 500);
        }
    }
} 