<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Certificate extends Model
{
    protected $table = "certificates";
    public $timestamps = false;
    protected $guarded = ['id'];

    // Certificate status constants
    const STATUS_PENDING = 'pending';
    const STATUS_GENERATED = 'generated';
    const STATUS_FAILED = 'failed';

    // Certificate types
    const TYPE_QUIZ = 'quiz';
    const TYPE_COURSE = 'course';
    const TYPE_BUNDLE = 'bundle';
    const TYPE_ACHIEVEMENT = 'achievement';

    protected $casts = [
        'created_at' => 'integer',
        'generated_at' => 'integer',
        'metadata' => 'array',
    ];

    public function quiz()
    {
        return $this->hasOne('App\Models\Quiz', 'id', 'quiz_id');
    }

    public function student()
    {
        return $this->hasOne('App\User', 'id', 'student_id');
    }

    public function quizzesResult()
    {
        return $this->hasOne('App\Models\QuizzesResult', 'id', 'quiz_result_id');
    }

    public function webinar()
    {
        return $this->belongsTo('App\Models\Webinar', 'webinar_id', 'id');
    }

    public function bundle()
    {
        return $this->belongsTo('App\Models\Bundle', 'bundle_id', 'id');
    }

    public function template()
    {
        return $this->belongsTo('App\Models\CertificateTemplate', 'template_id', 'id');
    }

    /**
     * Get certificate image URL
     */
    public function getImageUrlAttribute()
    {
        if ($this->image_path) {
            return url($this->image_path);
        }
        return url("store/certificates/certificate_{$this->id}.png");
    }

    /**
     * Get certificate HTML content
     */
    public function getHtmlContentAttribute()
    {
        return $this->metadata['html_content'] ?? null;
    }

    /**
     * Set certificate HTML content
     */
    public function setHtmlContentAttribute($value)
    {
        $metadata = $this->metadata ?? [];
        $metadata['html_content'] = $value;
        $this->metadata = $metadata;
    }

    /**
     * Get certificate QR code
     */
    public function getQrCodeAttribute()
    {
        return $this->metadata['qr_code'] ?? null;
    }

    /**
     * Set certificate QR code
     */
    public function setQrCodeAttribute($value)
    {
        $metadata = $this->metadata ?? [];
        $metadata['qr_code'] = $value;
        $this->metadata = $metadata;
    }

    /**
     * Check if certificate is generated
     */
    public function isGenerated()
    {
        return $this->status === self::STATUS_GENERATED && file_exists(public_path($this->image_path));
    }

    /**
     * Get certificate verification URL
     */
    public function getVerificationUrlAttribute()
    {
        return url("/certificate_validation/{$this->id}");
    }

    /**
     * Scope for generated certificates
     */
    public function scopeGenerated($query)
    {
        return $query->where('status', self::STATUS_GENERATED);
    }

    /**
     * Scope for pending certificates
     */
    public function scopePending($query)
    {
        return $query->where('status', self::STATUS_PENDING);
    }

    /**
     * Scope for failed certificates
     */
    public function scopeFailed($query)
    {
        return $query->where('status', self::STATUS_FAILED);
    }

    /**
     * Get formatted created date
     */
    public function getFormattedCreatedAtAttribute()
    {
        return $this->created_at ? date('Y-m-d H:i:s', $this->created_at) : null;
    }

    /**
     * Get formatted generated date
     */
    public function getFormattedGeneratedAtAttribute()
    {
        return $this->generated_at ? date('Y-m-d H:i:s', $this->generated_at) : null;
    }
}
