import 'dart:convert';
import 'dart:io';
import 'package:http/http.dart' as http;
import 'package:path_provider/path_provider.dart';
import 'package:webinar/app/models/certificate_model.dart';

import '../../../common/enums/error_enum.dart';
import '../../../common/utils/constants.dart';
import '../../../common/utils/error_handler.dart';
import '../../../common/utils/http_handler.dart';

class CertificateService {
  /// Fetches the user's achievement certificates (list)
  static Future<List<CertificateModel>> getAchievements() async {
    List<CertificateModel> data = [];
    try {
      String url = '${Constants.baseUrl}panel/certificates/achievements';
      final res = await httpGetWithToken(url);
      var jsonResponse = jsonDecode(res.body);
      if (jsonResponse['success']) {
        jsonResponse['data'].forEach((json) {
          data.add(CertificateModel.fromJson(json));
        });
        return data;
      } else {
        ErrorHandler().showError(ErrorEnum.error, jsonResponse);
        return data;
      }
    } catch (e) {
      return data;
    }
  }

  /// Fetches the user's completion certificates (list)
  static Future<List<CertificateModel>> getCompletion() async {
    List<CertificateModel> data = [];
    try {
      String url = '${Constants.baseUrl}panel/webinars/certificates';
      final res = await httpGetWithToken(url);
      var jsonResponse = jsonDecode(res.body);
      if (jsonResponse['success']) {
        jsonResponse['data']['certificates'].forEach((json) {
          data.add(CertificateModel.fromJson(json));
        });
        return data;
      } else {
        ErrorHandler().showError(ErrorEnum.error, jsonResponse);
        return data;
      }
    } catch (e) {
      return data;
    }
  }

  /// Fetches the user's created class certificates (list)
  static Future<List<CertificateModel>> getClassData() async {
    List<CertificateModel> data = [];
    try {
      String url = '${Constants.baseUrl}panel/certificates/created';
      final res = await httpGetWithToken(url);
      var jsonResponse = jsonDecode(res.body);
      if (jsonResponse['success']) {
        jsonResponse['data']['certificates'].forEach((json) {
          data.add(CertificateModel.fromJson(json));
        });
        return data;
      } else {
        ErrorHandler().showError(ErrorEnum.error, jsonResponse);
        return data;
      }
    } catch (e) {
      return data;
    }
  }

  /// Download certificate image from the backend (new API)
  /// Returns the local file path, or null on failure
  static Future<String?> downloadCertificateImage(int certificateId, int userId, {String type = 'quiz'}) async {
    try {
      // Use your actual domain here!
      String url = 'https://neft.sa/api/development/certificates/generate-image?certificate_id=$certificateId&user_id=$userId&type=$type';
      final response = await http.get(Uri.parse(url));
      if (response.statusCode == 200) {
        Directory appDir = await getApplicationDocumentsDirectory();
        Directory certificatesDir = Directory('${appDir.path}/certificates');
        if (!await certificatesDir.exists()) {
          await certificatesDir.create(recursive: true);
        }
        String imagePath = '${certificatesDir.path}/certificate_$certificateId.png';
        File imageFile = File(imagePath);
        await imageFile.writeAsBytes(response.bodyBytes);
        return imagePath;
      }
      return null;
    } catch (e) {
      print('Error downloading certificate image: $e');
      return null;
    }
  }

  /// Get the direct URL to the certificate image (for use with Image.network or CachedNetworkImage)
  static String getCertificateImageUrl(int certificateId) {
    return 'https://neft.sa/store/certificates/certificate_$certificateId.png';
  }

  /// Safely download certificate image with authentication and error handling
  /// Returns the local file path, or null on failure
  static Future<String?> downloadCertificateWithAuth(String imageUrl, String? token) async {
    try {
      // Check if we already have a cached version
      Directory appDir = await getApplicationDocumentsDirectory();
      Directory certificatesDir = Directory('${appDir.path}/certificates');
      if (!await certificatesDir.exists()) {
        await certificatesDir.create(recursive: true);
      }

      // Create a unique filename based on the URL
      String fileName = 'certificate_${DateTime.now().millisecondsSinceEpoch}.png';
      String imagePath = '${certificatesDir.path}/$fileName';
      File imageFile = File(imagePath);

      // Prepare headers
      Map<String, String> headers = {
        'Accept': 'image/*',
        'Content-Type': 'application/json',
        'x-api-key': Constants.apiKey,
      };

      // Add authorization header if token is available
      if (token != null && token.isNotEmpty) {
        headers['Authorization'] = 'Bearer $token';
      }

      // Download the image
      final response = await http.get(
        Uri.parse(imageUrl),
        headers: headers,
      );

      // Check if the response is successful and contains image data
      if (response.statusCode == 200 && response.bodyBytes.isNotEmpty) {
        // Verify it's actually an image by checking the first few bytes
        if (response.bodyBytes.length > 4) {
          // Check for common image file signatures
          List<int> bytes = response.bodyBytes;
          bool isImage = false;
          
          // PNG signature: 89 50 4E 47
          if (bytes.length >= 4 && bytes[0] == 0x89 && bytes[1] == 0x50 && bytes[2] == 0x4E && bytes[3] == 0x47) {
            isImage = true;
          }
          // JPEG signature: FF D8 FF
          else if (bytes.length >= 3 && bytes[0] == 0xFF && bytes[1] == 0xD8 && bytes[2] == 0xFF) {
            isImage = true;
          }
          // GIF signature: 47 49 46
          else if (bytes.length >= 3 && bytes[0] == 0x47 && bytes[1] == 0x49 && bytes[2] == 0x46) {
            isImage = true;
          }

          if (isImage) {
            await imageFile.writeAsBytes(response.bodyBytes);
            return imagePath;
          } else {
            print('Response is not a valid image file');
            return null;
          }
        } else {
          print('Response body is empty or too small');
          return null;
        }
      } else {
        print('HTTP Error: ${response.statusCode}');
        print('Response body: ${response.body}');
        return null;
      }
    } catch (e) {
      print('Error downloading certificate image: $e');
      return null;
    }
  }

  /// Download certificate using the existing download manager
  static Future<String?> downloadCertificate(int certificateId) async {
    try {
      String url = '${Constants.baseUrl}panel/certificates/download/$certificateId';
      final res = await httpGetWithToken(url);
      
      if (res.statusCode == 200) {
        Directory appDir = await getApplicationDocumentsDirectory();
        Directory certificatesDir = Directory('${appDir.path}/certificates');
        if (!await certificatesDir.exists()) {
          await certificatesDir.create(recursive: true);
        }
        
        String imagePath = '${certificatesDir.path}/certificate_$certificateId.png';
        File imageFile = File(imagePath);
        await imageFile.writeAsBytes(res.bodyBytes);
        return imagePath;
      }
      return null;
    } catch (e) {
      print('Error downloading certificate: $e');
      return null;
    }
  }
} 