import 'dart:io';
import 'package:flutter/material.dart';
import 'package:http/http.dart' as http;

class DebugImageWidget extends StatefulWidget {
  final String imageUrl;
  final Map<String, String>? headers;
  final double? width;
  final double? height;
  final BoxFit fit;

  const DebugImageWidget({
    Key? key,
    required this.imageUrl,
    this.headers,
    this.width,
    this.height,
    this.fit = BoxFit.cover,
  }) : super(key: key);

  @override
  State<DebugImageWidget> createState() => _DebugImageWidgetState();
}

class _DebugImageWidgetState extends State<DebugImageWidget> {
  bool _isLoading = true;
  String? _error;
  String? _localPath;
  Map<String, dynamic>? _debugInfo;

  @override
  void initState() {
    super.initState();
    _loadImage();
  }

  Future<void> _loadImage() async {
    setState(() {
      _isLoading = true;
      _error = null;
      _debugInfo = null;
    });

    try {
      print('Loading image from: ${widget.imageUrl}');
      print('Headers: ${widget.headers}');

      final response = await http.get(
        Uri.parse(widget.imageUrl),
        headers: widget.headers,
      );

      _debugInfo = {
        'statusCode': response.statusCode,
        'contentType': response.headers['content-type'],
        'contentLength': response.bodyBytes.length,
        'responseBody': response.body.substring(0, response.body.length > 200 ? 200 : response.body.length),
      };

      print('Response status: ${response.statusCode}');
      print('Content-Type: ${response.headers['content-type']}');
      print('Content-Length: ${response.bodyBytes.length}');

      if (response.statusCode == 200 && response.bodyBytes.isNotEmpty) {
        // Check if it's actually an image
        List<int> bytes = response.bodyBytes;
        bool isImage = false;
        String imageType = 'Unknown';

        // Check for common image file signatures
        if (bytes.length >= 4 && bytes[0] == 0x89 && bytes[1] == 0x50 && bytes[2] == 0x4E && bytes[3] == 0x47) {
          isImage = true;
          imageType = 'PNG';
        } else if (bytes.length >= 3 && bytes[0] == 0xFF && bytes[1] == 0xD8 && bytes[2] == 0xFF) {
          isImage = true;
          imageType = 'JPEG';
        } else if (bytes.length >= 3 && bytes[0] == 0x47 && bytes[1] == 0x49 && bytes[2] == 0x46) {
          isImage = true;
          imageType = 'GIF';
        }

        _debugInfo!['isImage'] = isImage;
        _debugInfo!['imageType'] = imageType;

        if (isImage) {
          // Save to local file
          final directory = await Directory.systemTemp.createTemp('debug_images');
          final file = File('${directory.path}/debug_image_${DateTime.now().millisecondsSinceEpoch}.png');
          await file.writeAsBytes(response.bodyBytes);
          
          setState(() {
            _localPath = file.path;
            _isLoading = false;
          });
        } else {
          setState(() {
            _error = 'Response is not a valid image (detected as: $imageType)';
            _isLoading = false;
          });
        }
      } else {
        setState(() {
          _error = 'HTTP Error: ${response.statusCode}';
          _isLoading = false;
        });
      }
    } catch (e) {
      setState(() {
        _error = 'Exception: $e';
        _isLoading = false;
      });
    }
  }

  @override
  Widget build(BuildContext context) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Container(
          width: widget.width,
          height: widget.height,
          decoration: BoxDecoration(
            border: Border.all(color: Colors.grey),
            borderRadius: BorderRadius.circular(8),
          ),
          child: _isLoading
              ? Center(
                  child: Column(
                    mainAxisAlignment: MainAxisAlignment.center,
                    children: [
                      CircularProgressIndicator(),
                      SizedBox(height: 8),
                      Text('Loading...'),
                    ],
                  ),
                )
              : _error != null
                  ? Center(
                      child: Column(
                        mainAxisAlignment: MainAxisAlignment.center,
                        children: [
                          Icon(Icons.error_outline, color: Colors.red, size: 48),
                          SizedBox(height: 8),
                          Text(
                            'Error',
                            style: TextStyle(color: Colors.red, fontWeight: FontWeight.bold),
                          ),
                          SizedBox(height: 4),
                          Text(
                            _error!,
                            style: TextStyle(fontSize: 12),
                            textAlign: TextAlign.center,
                          ),
                        ],
                      ),
                    )
                  : _localPath != null
                      ? ClipRRect(
                          borderRadius: BorderRadius.circular(8),
                          child: Image.file(
                            File(_localPath!),
                            width: widget.width,
                            height: widget.height,
                            fit: widget.fit,
                            errorBuilder: (context, error, stackTrace) {
                              return Center(
                                child: Column(
                                  mainAxisAlignment: MainAxisAlignment.center,
                                  children: [
                                    Icon(Icons.broken_image, color: Colors.orange, size: 48),
                                    SizedBox(height: 8),
                                    Text('Failed to display image'),
                                  ],
                                ),
                              );
                            },
                          ),
                        )
                      : Center(
                          child: Text('No image loaded'),
                        ),
        ),
        if (_debugInfo != null) ...[
          SizedBox(height: 16),
          Container(
            padding: EdgeInsets.all(12),
            decoration: BoxDecoration(
              color: Colors.grey[100],
              borderRadius: BorderRadius.circular(8),
            ),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(
                  'Debug Information:',
                  style: TextStyle(fontWeight: FontWeight.bold),
                ),
                SizedBox(height: 8),
                ..._debugInfo!.entries.map((entry) => Padding(
                  padding: EdgeInsets.only(bottom: 4),
                  child: Text(
                    '${entry.key}: ${entry.value}',
                    style: TextStyle(fontSize: 12, fontFamily: 'monospace'),
                  ),
                )),
              ],
            ),
          ),
        ],
        SizedBox(height: 8),
        ElevatedButton(
          onPressed: _loadImage,
          child: Text('Reload Image'),
        ),
      ],
    );
  }
} 