<?php

/**
 * Test script to verify certification matching between certifications.id_no/govt_id and users.student_id
 * Run this script to test the matching functionality
 */

require_once 'vendor/autoload.php';

use App\Models\Certification;
use App\Models\User;

// Test the matching functionality
echo "Testing certification matching (ID_NO and GOVT_ID)...\n\n";

// Get some sample data
$certifications = Certification::take(10)->get();
$users = User::whereNotNull('student_id')->take(10)->get();

echo "Sample certifications:\n";
foreach ($certifications as $cert) {
    echo "ID: {$cert->Id}, Name: {$cert->name}, ID_NO: {$cert->id_no}, GOVT_ID: {$cert->govt_id}\n";
}

echo "\nSample users with student_id:\n";
foreach ($users as $user) {
    echo "ID: {$user->id}, Name: {$user->full_name}, Student_ID: {$user->student_id}\n";
}

echo "\nTesting matching (ID_NO OR GOVT_ID):\n";
foreach ($certifications as $cert) {
    $matchingUser = User::where('student_id', $cert->id_no)
                       ->orWhere('student_id', $cert->govt_id)
                       ->first();
    if ($matchingUser) {
        $matchType = '';
        if ($matchingUser->student_id == $cert->id_no) {
            $matchType = 'ID_NO';
        } elseif ($matchingUser->student_id == $cert->govt_id) {
            $matchType = 'GOVT_ID';
        }
        echo "✓ Certification {$cert->Id} ({$matchType}: {$matchingUser->student_id}) matches User {$matchingUser->id} (Student_ID: {$matchingUser->student_id})\n";
    } else {
        echo "✗ Certification {$cert->Id} (ID_NO: {$cert->id_no}, GOVT_ID: {$cert->govt_id}) has no matching user\n";
    }
}

echo "\nTesting reverse matching (Student_ID matches ID_NO OR GOVT_ID):\n";
foreach ($users as $user) {
    if ($user->student_id) {
        $matchingCert = Certification::where('id_no', $user->student_id)
                                   ->orWhere('govt_id', $user->student_id)
                                   ->first();
        if ($matchingCert) {
            $matchType = '';
            if ($matchingCert->id_no == $user->student_id) {
                $matchType = 'ID_NO';
            } elseif ($matchingCert->govt_id == $user->student_id) {
                $matchType = 'GOVT_ID';
            }
            echo "✓ User {$user->id} (Student_ID: {$user->student_id}) matches Certification {$matchingCert->Id} ({$matchType}: {$user->student_id})\n";
        } else {
            echo "✗ User {$user->id} (Student_ID: {$user->student_id}) has no matching certification\n";
        }
    }
}

echo "\nTesting specific matching scenarios:\n";

// Test ID_NO matching only
echo "\n1. Testing ID_NO matching only:\n";
$idNoMatches = Certification::whereNotNull('id_no')->take(3)->get();
foreach ($idNoMatches as $cert) {
    $matchingUser = User::where('student_id', $cert->id_no)->first();
    if ($matchingUser) {
        echo "✓ ID_NO match: Certification {$cert->Id} (ID_NO: {$cert->id_no}) matches User {$matchingUser->id}\n";
    } else {
        echo "✗ ID_NO match: Certification {$cert->Id} (ID_NO: {$cert->id_no}) has no matching user\n";
    }
}

// Test GOVT_ID matching only
echo "\n2. Testing GOVT_ID matching only:\n";
$govtIdMatches = Certification::whereNotNull('govt_id')->take(3)->get();
foreach ($govtIdMatches as $cert) {
    $matchingUser = User::where('student_id', $cert->govt_id)->first();
    if ($matchingUser) {
        echo "✓ GOVT_ID match: Certification {$cert->Id} (GOVT_ID: {$cert->govt_id}) matches User {$matchingUser->id}\n";
    } else {
        echo "✗ GOVT_ID match: Certification {$cert->Id} (GOVT_ID: {$cert->govt_id}) has no matching user\n";
    }
}

// Test combined matching
echo "\n3. Testing combined matching (ID_NO OR GOVT_ID):\n";
$combinedMatches = Certification::take(5)->get();
foreach ($combinedMatches as $cert) {
    $matchingUser = User::where(function($query) use ($cert) {
        $query->where('student_id', $cert->id_no)
              ->orWhere('student_id', $cert->govt_id);
    })->first();
    
    if ($matchingUser) {
        $matchTypes = [];
        if ($matchingUser->student_id == $cert->id_no) {
            $matchTypes[] = 'ID_NO';
        }
        if ($matchingUser->student_id == $cert->govt_id) {
            $matchTypes[] = 'GOVT_ID';
        }
        echo "✓ Combined match: Certification {$cert->Id} matches User {$matchingUser->id} via: " . implode(', ', $matchTypes) . "\n";
    } else {
        echo "✗ Combined match: Certification {$cert->Id} has no matching user\n";
    }
}

echo "\nTest completed!\n"; 