<?php

/* =======================================
 * 	JobCareer Functions - Child Theme
 */

// Hide WordPress Admin Bar for non-admin users
add_action('after_setup_theme', 'jobcareer_child_hide_admin_bar_for_non_admins');
function jobcareer_child_hide_admin_bar_for_non_admins() {
    // If user is not an administrator, hide the admin bar
    if (!current_user_can('administrator') && !is_admin()) {
        show_admin_bar(false);
    }
}

// Remove admin bar for non-admins on frontend (additional enforcement)
add_filter('show_admin_bar', 'jobcareer_child_remove_admin_bar_for_non_admins');
function jobcareer_child_remove_admin_bar_for_non_admins($show_admin_bar) {
    // Only show admin bar for administrators
    if (!current_user_can('administrator')) {
        return false;
    }
    return $show_admin_bar;
}

// Override user meta to prevent admin bar from showing for non-admins
add_action('init', 'jobcareer_child_force_hide_admin_bar_non_admins', 9);
function jobcareer_child_force_hide_admin_bar_non_admins() {
    if (is_user_logged_in() && !current_user_can('administrator')) {
        add_filter('get_user_metadata', 'jobcareer_child_force_admin_bar_false', 10, 4);
    }
}

function jobcareer_child_force_admin_bar_false($null, $object_id, $meta_key, $single) {
    if ($meta_key == 'show_admin_bar_front') {
        return 'false';
    }
    return $null;
}

// Override specialisms dropdown to show only parent specialisms for registration forms
if (!function_exists('cs_get_specialisms_dropdown_parent_only')) {

    function cs_get_specialisms_dropdown_parent_only($name, $id, $user_id = '', $class = '', $required_status = 'false') {
        global $cs_form_fields2, $post;
        $output = '';
        $cs_spec_args = array(
            'orderby' => 'name',
            'order' => 'ASC',
            'fields' => 'all',
            'slug' => '',
            'hide_empty' => false,
            'parent' => 0, // Only show parent specialisms (no children)
        );
        $terms = get_terms('specialisms', $cs_spec_args);
        if (!empty($terms)) {

            $please_select_specialisms_label = esc_html__('Please Select specialism', 'jobhunt');
            $please_select_specialisms_label = apply_filters('jobhunt_replace_please_select_specialism_to_category', $please_select_specialisms_label);

            $cs_selected_specs = get_user_meta($user_id, $name, true);
            $specialisms_option = '';
            foreach ($terms as $term) {
                $cs_selected = '';
                if (is_array($cs_selected_specs) && in_array($term->slug, $cs_selected_specs)) {
                    $cs_selected = ' selected="selected"';
                }
                $specialisms_option .= '<option' . $cs_selected . ' value="' . esc_attr($term->slug) . '">' . $term->name . '</option>';
            }
            $cs_opt_array = array(
                'cust_id' => $id,
                'cust_name' => $name . '[]',
                'std' => '',
                'desc' => '',
                'return' => true,
                'extra_atr' => 'data-placeholder="' . $please_select_specialisms_label . '"',
                'classes' => $class,
                'options' => $specialisms_option,
                'options_markup' => true,
                'hint_text' => '',
            );
            if (isset($required_status) && $required_status == true) {
                $cs_opt_array['required'] = 'yes';
            }
            $output .= $cs_form_fields2->cs_form_multiselect_render($cs_opt_array);
        } else {
            $no_specialisms_available_label = esc_html__('There are no specialisms available.', 'jobhunt');
            $no_specialisms_available_label = apply_filters('jobhunt_replace_no_specialisms_available', $no_specialisms_available_label);
            $output .= $no_specialisms_available_label;
        }
        return $output;
    }

}

// Simple approach: Filter the terms query to show only parent specialisms in registration forms
add_filter('get_terms_args', 'jobcareer_child_filter_specialisms_for_registration', 10, 2);
function jobcareer_child_filter_specialisms_for_registration($args, $taxonomies) {
    try {
        // Only apply this filter for specialisms taxonomy
        if (in_array('specialisms', $taxonomies)) {
            // Simple check: if we're on a page with 'register' or 'login' in the URL or in the template
            $current_url = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '';
            $is_registration_page = (
                strpos($current_url, 'register') !== false || 
                strpos($current_url, 'login') !== false ||
                strpos($current_url, 'signup') !== false ||
                strpos($current_url, 'account') !== false
            );
            
            // Also check the backtrace for registration templates
            $backtrace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 8);
            $is_registration_template = false;
            
            foreach ($backtrace as $trace) {
                if (isset($trace['file'])) {
                    $file_path = $trace['file'];
                    if (strpos($file_path, 'register') !== false || 
                        strpos($file_path, 'login') !== false ||
                        strpos($file_path, 'shortcodes') !== false) {
                        $is_registration_template = true;
                        break;
                    }
                }
            }
            
            // If it's a registration context, only show parent specialisms
            if ($is_registration_page || $is_registration_template) {
                $args['parent'] = 0; // Only show parent specialisms
            }
        }
    } catch (Exception $e) {
        // If there's any error, just return the original args
        error_log('JobCareer Child Theme Error: ' . $e->getMessage());
    }
    return $args;
}

// Custom registration form field modifications
add_action('init', 'jobcareer_child_customize_registration_fields', 20);

function jobcareer_child_customize_registration_fields() {
    // Add custom JavaScript for form validation
    add_action('wp_footer', 'jobcareer_child_registration_validation_script');
}

// Custom JavaScript for registration form validation
function jobcareer_child_registration_validation_script() {
    // Load on all pages to ensure it works
    ?>
    <script type="text/javascript">
    jQuery(document).ready(function($) {
        // Wait a bit for dynamic content to load
        setTimeout(function() {
            
            // ========================================
            // FORM VALIDATION & BUTTON STATE CONTROL
            // ========================================
            
            // Function to check form validity and toggle submit button
            function checkFormValidityAndToggleButton(form) {
                // ONLY target submit buttons within REGISTRATION forms, exclude profile update buttons
                var submitBtn = form.find('.acc-submit:not(.cs-section-update)');
                if (submitBtn.length === 0) return;
                
                // Custom validation instead of relying on HTML5 validation
                var isValid = true;
                var invalidFields = [];
                
                // Check each required field individually
                form.find('input[required], select[required], textarea[required]').each(function() {
                    var field = $(this);
                    var fieldName = field.attr('name') || field.attr('id') || 'unknown';
                    var fieldValue = field.val();
                    var fieldValid = true;
                    
                    // Custom validation for username field
                    if (fieldName.indexOf('user_login') !== -1) {
                        if (fieldValue.length < 3) {
                            fieldValid = false;
                        } else {
                            var arabicRegex = /^[\u0600-\u06FF\u0750-\u077F\u08A0-\u08FF\uFB50-\uFDFF\uFE70-\uFEFFa-zA-Z\s]+$/;
                            fieldValid = arabicRegex.test(fieldValue);
                        }
                    }
                    // Custom validation for email field
                    else if (fieldName.indexOf('email') !== -1) {
                        var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                        fieldValid = emailRegex.test(fieldValue);
                    }
                    // Custom validation for phone field
                    else if (fieldName.indexOf('phone') !== -1) {
                        var cleanPhone = fieldValue.replace(/\D/g, '');
                        fieldValid = cleanPhone.length === 9 && /^5[0-9]{8}$/.test(cleanPhone);
                    }
                    // For other fields, use HTML5 validation
                    else {
                        fieldValid = field[0].checkValidity();
                    }
                    
                    console.log('Field check:', fieldName, 'Value:', fieldValue, 'Valid:', fieldValid);
                    
                    if (!fieldValid) {
                        isValid = false;
                        invalidFields.push(fieldName);
                    }
                });
                
                console.log('Custom form validity check:', isValid, 'Invalid fields:', invalidFields);
                
                // Also check for any visible error messages
                var hasErrors = form.find('.field-error').length > 0;
                console.log('Has errors:', hasErrors);
                
                // Check if terms and conditions checkbox is checked (if it exists)
                var termsCheckbox = form.find('input[id*="jobhunt_check_terms"]');
                var termsChecked = true; // Default to true if checkbox doesn't exist
                if (termsCheckbox.length > 0) {
                    termsChecked = termsCheckbox.is(':checked');
                    console.log('Terms checkbox found:', termsCheckbox.length, 'Checked:', termsChecked);
                } else {
                    console.log('No terms checkbox found');
                }
                
                // Enable button only if form is valid, no errors, and terms are checked
                if (isValid && !hasErrors && termsChecked) {
                    console.log('Enabling submit button');
                    submitBtn.prop('disabled', false);
                    submitBtn.removeClass('submit-disabled');
                    submitBtn.addClass('submit-enabled');
                } else {
                    console.log('Disabling submit button - isValid:', isValid, 'hasErrors:', hasErrors, 'termsChecked:', termsChecked);
                    submitBtn.prop('disabled', true);
                    submitBtn.removeClass('submit-enabled');
                    submitBtn.addClass('submit-disabled');
                }
            }
            
            // Function to make form fields required
            function makeFieldsRequired(form) {
                // Username field
                form.find('input[id*="user_login"], input[name*="user_login"], input[placeholder*="Username"]').attr('required', 'required');
                
                // Email field
                form.find('input[type="email"], input[id*="email"], input[name*="email"]').attr('required', 'required');
                
                // Password field (if visible)
                form.find('input[type="password"], input[id*="password"]').attr('required', 'required');
                
                // Phone number field
                form.find('input[id*="phone"], input[name*="phone"], input[placeholder*="Phone"]').attr('required', 'required');
                
                // Employer specific fields
                form.find('input[id*="organization_name"], input[name*="organization_name"]').attr('required', 'required');
                
                // Company Number field (Employer) - only if field exists
                var companyNumberField = form.find('input[id*="company_number"], input[name*="company_number"]');
                if (companyNumberField.length > 0) {
                    companyNumberField.attr('required', 'required');
                }
                
                // Company Address field (Employer) - only if field exists
                var companyAddressField = form.find('select[id*="company_address"], select[name*="company_address"]');
                if (companyAddressField.length > 0) {
                    companyAddressField.attr('required', 'required');
                }
                
                // Specialism dropdown (Candidate)
                form.find('select[id*="candidate_specialisms"]').attr('required', 'required');
                
                // Specialism field (Employer)
                form.find('input[id*="employer_specialisms"]').attr('required', 'required');
                
                // Terms and Conditions checkbox
                form.find('input[id*="jobhunt_check_terms"]').attr('required', 'required');
            }
            
            // ========================================
            // INLINE FIELD VALIDATION WITH ERROR MESSAGES
            // ========================================
            
            // Function to show field error message
            function showFieldError(input, message) {
                var errorDiv = input.siblings('.field-error');
                if (errorDiv.length === 0) {
                    errorDiv = $('<div class="field-error" style="color: red; font-size: 12px; margin-top: 5px;"></div>');
                    input.after(errorDiv);
                }
                errorDiv.text(message).show();
                input.addClass('error-field');
            }
            
            // Function to hide field error message
            function hideFieldError(input) {
                var errorDiv = input.siblings('.field-error');
                errorDiv.hide().remove();
                input.removeClass('error-field');
                
                // Also check parent containers for phone fields
                var phoneContainer = input.closest('.phone-field-container');
                if (phoneContainer.length) {
                    phoneContainer.find('.field-error').hide().remove();
                }
                
                var fieldHolder = input.closest('.field-holder');
                if (fieldHolder.length) {
                    fieldHolder.find('.field-error').hide().remove();
                }
            }
            
            // Function to show phone field error (for phone container)
            function showPhoneFieldError(container, message) {
                var errorDiv = container.find('.field-error');
                if (errorDiv.length === 0) {
                    errorDiv = $('<div class="field-error" style="color: red; font-size: 12px; margin-top: 5px;"></div>');
                    container.append(errorDiv);
                }
                errorDiv.text(message).show();
                container.find('input').addClass('error-field');
            }
            
            // Full Name validation (letters and spaces only, minimum 3 characters)
            $(document).on('blur keyup input', 'input[id*="user_login"], input[name*="user_login"], input[placeholder*="Username"]', function() {
                var input = $(this);
                var value = $.trim(input.val());
                var form = input.closest('form[id*="wp_signup_form"]');
                
                // Always hide error first
                hideFieldError(input);
                
                if (value.length > 0) {
                    console.log('Validating username:', value);
                    if (value.length < 3) {
                        console.log('Username too short');
                        showFieldError(input, 'Full name must be at least 3 characters');
                        if (form.length) checkFormValidityAndToggleButton(form);
                        return false;
                    }
                    var arabicRegex = /^[\u0600-\u06FF\u0750-\u077F\u08A0-\u08FF\uFB50-\uFDFF\uFE70-\uFEFFa-zA-Z\s]+$/;
                    console.log('Testing regex:', arabicRegex.test(value));
                    if (!arabicRegex.test(value)) {
                        console.log('Username contains invalid characters');
                        showFieldError(input, 'Full name can only contain letters and spaces');
                        if (form.length) checkFormValidityAndToggleButton(form);
                        return false;
                    }
                    console.log('Username validation passed');
                    // If we reach here, validation passed - hide any existing errors
                    hideFieldError(input);
                }
                if (form.length) checkFormValidityAndToggleButton(form);
                return true;
            });
            
            // Email validation for candidate registration only
            $(document).on('blur keyup input', 'input[type="email"], input[id*="email"], input[name*="email"]', function() {
                var input = $(this);
                var value = $.trim(input.val());
                var form = input.closest('form[id*="wp_signup_form"]');
                
                // Always hide error first
                hideFieldError(input);
                
                if (value.length > 0) {
                    var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                    if (!emailRegex.test(value)) {
                        showFieldError(input, 'Please enter a valid email address');
                        if (form.length) checkFormValidityAndToggleButton(form);
                        return false;
                    }
                    // If email format is valid, check for duplicates
                    checkDuplicateEmail(value, input);
                }
                if (form.length) checkFormValidityAndToggleButton(form);
                return true;
            });
            
            // Function to check duplicate email
            function checkDuplicateEmail(email, input) {
                var form = input.closest('form[id*="wp_signup_form"]');
                $.ajax({
                    url: '<?php echo admin_url('admin-ajax.php'); ?>',
                    type: 'POST',
                    data: {
                        action: 'jobcareer_check_duplicate_email',
                        email: email,
                        nonce: '<?php echo wp_create_nonce("email_check_nonce"); ?>'
                    },
                    success: function(response) {
                        if (response.success === false) {
                            showFieldError(input, 'This email address is already registered. Please use a different email.');
                        } else {
                            // Email is available - hide any errors and remove error styling
                            hideFieldError(input);
                            input.removeClass('error-field');
                        }
                        if (form.length) {
                            setTimeout(function() {
                                checkFormValidityAndToggleButton(form);
                            }, 100);
                        }
                    },
                    error: function() {
                        console.log('Error checking duplicate email');
                        // On error, assume email is valid and hide errors
                        hideFieldError(input);
                        input.removeClass('error-field');
                        if (form.length) {
                            setTimeout(function() {
                                checkFormValidityAndToggleButton(form);
                            }, 100);
                        }
                    }
                });
            }
            
            // Phone number validation (Saudi format: starts with 5, exactly 9 digits)
            $(document).on('blur keyup input', 'input[id*="phone"], input[name*="phone"], input[placeholder*="Phone"]', function() {
                var input = $(this);
                var value = $.trim(input.val());
                var phoneContainer = input.closest('.phone-field-container');
                var fieldHolder = phoneContainer.length ? phoneContainer.parent() : input.closest('.field-holder');
                var form = input.closest('form[id*="wp_signup_form"]');
                
                // Always hide error first
                if (fieldHolder.find('.field-error').length) {
                    fieldHolder.find('.field-error').remove();
                }
                input.removeClass('error-field');
                
                if (value.length > 0) {
                    var cleanValue = value.replace(/\D/g, '');
                    if (cleanValue.length !== 9) {
                        showPhoneFieldError(fieldHolder, 'Phone number must be exactly 9 digits');
                        if (form.length) checkFormValidityAndToggleButton(form);
                        return false;
                    }
                    if (!cleanValue.startsWith('5')) {
                        showPhoneFieldError(fieldHolder, 'Phone number must start with 5');
                        if (form.length) checkFormValidityAndToggleButton(form);
                        return false;
                    }
                    // If we reach here, validation passed - ensure no errors are shown
                    if (fieldHolder.find('.field-error').length) {
                        fieldHolder.find('.field-error').remove();
                    }
                    input.removeClass('error-field');
                }
                if (form.length) checkFormValidityAndToggleButton(form);
                return true;
            });
            
            // Custom phone number formatting
            $(document).on('input', 'input[id*="phone"], input[name*="phone"], input[placeholder*="Phone"]', function() {
                var input = $(this);
                var value = input.val().replace(/\D/g, ''); // Remove non-digits
                if (value.length > 9) {
                    value = value.substring(0, 9);
                }
                input.val(value);
            });
            
            // Company Number validation (exactly 10 digits, numbers only)
            $(document).on('blur keyup input', 'input[id*="company_number"], input[name*="company_number"]', function() {
                var input = $(this);
                var value = $.trim(input.val());
                var form = input.closest('form[id*="wp_signup_form"]');
                
                // Always hide error first
                hideFieldError(input);
                
                if (value.length > 0) {
                    // Remove any non-digit characters
                    var cleanValue = value.replace(/\D/g, '');
                    if (cleanValue.length !== 10) {
                        showFieldError(input, 'Company number must be exactly 10 digits');
                        if (form.length) checkFormValidityAndToggleButton(form);
                        return false;
                    }
                    // If we reach here, validation passed
                    hideFieldError(input);
                }
                if (form.length) checkFormValidityAndToggleButton(form);
                return true;
            });
            
            // Company Number formatting (numbers only, max 10 digits)
            $(document).on('input', 'input[id*="company_number"], input[name*="company_number"]', function() {
                var input = $(this);
                var value = input.val().replace(/\D/g, ''); // Remove non-digits
                if (value.length > 10) {
                    value = value.substring(0, 10);
                }
                input.val(value);
            });
            
            // Company Address validation (dropdown selection)
            $(document).on('change', 'select[id*="company_address"], select[name*="company_address"]', function() {
                var select = $(this);
                var value = select.val();
                var form = select.closest('form[id*="wp_signup_form"]');
                
                // Always hide error first
                hideFieldError(select);
                
                if (value === '' || value === null) {
                    showFieldError(select, 'Please select a region');
                    if (form.length) checkFormValidityAndToggleButton(form);
                    return false;
                }
                // If we reach here, validation passed
                hideFieldError(select);
                if (form.length) checkFormValidityAndToggleButton(form);
                return true;
            });
            
            // Initialize all registration forms
            $('form[id*="wp_signup_form"]').each(function() {
                var form = $(this);
                
                // Make fields required
                makeFieldsRequired(form);
                
                // Initial check
                checkFormValidityAndToggleButton(form);
                
                // Listen to input events for real-time validation
                form.on('input keyup change', 'input, select, textarea', function() {
                    var input = $(this);
                    var value = $.trim(input.val());
                    
                    // Real-time validation for specific fields
                    if (input.is('input[id*="user_login"], input[name*="user_login"], input[placeholder*="Username"]')) {
                        console.log('Real-time validation for username:', value);
                        hideFieldError(input);
                        if (value.length > 0) {
                        if (value.length < 3) {
                            console.log('Real-time: Username too short');
                            showFieldError(input, 'Full name must be at least 3 characters');
                        } else {
                            var arabicRegex = /^[\u0600-\u06FF\u0750-\u077F\u08A0-\u08FF\uFB50-\uFDFF\uFE70-\uFEFFa-zA-Z\s]+$/;
                            console.log('Real-time: Testing regex:', arabicRegex.test(value));
                            if (!arabicRegex.test(value)) {
                                console.log('Real-time: Username contains invalid characters');
                                showFieldError(input, 'Full name can only contain letters and spaces');
                            } else {
                                console.log('Real-time: Username validation passed');
                            }
                        }
                        }
                    }
                    
                    if (input.is('input[type="email"], input[id*="email"], input[name*="email"]')) {
                        hideFieldError(input);
                        if (value.length > 0) {
                            var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                            if (!emailRegex.test(value)) {
                                showFieldError(input, 'Please enter a valid email address');
                            }
                        }
                    }
                    
                    if (input.is('input[id*="phone"], input[name*="phone"], input[placeholder*="Phone"]')) {
                        var phoneContainer = input.closest('.phone-field-container');
                        var fieldHolder = phoneContainer.length ? phoneContainer.parent() : input.closest('.field-holder');
                        
                        if (fieldHolder.find('.field-error').length) {
                            fieldHolder.find('.field-error').remove();
                        }
                        input.removeClass('error-field');
                        
                        if (value.length > 0) {
                            var cleanValue = value.replace(/\D/g, '');
                            if (cleanValue.length !== 9) {
                                showPhoneFieldError(fieldHolder, 'Phone number must be exactly 9 digits');
                            } else if (!cleanValue.startsWith('5')) {
                                showPhoneFieldError(fieldHolder, 'Phone number must start with 5');
                            }
                        }
                    }
                    
                    // Company Number real-time validation
                    if (input.is('input[id*="company_number"], input[name*="company_number"]')) {
                        hideFieldError(input);
                        if (value.length > 0) {
                            var cleanValue = value.replace(/\D/g, '');
                            if (cleanValue.length !== 10) {
                                showFieldError(input, 'Company number must be exactly 10 digits');
                            }
                        }
                    }
                    
                    // Company Address real-time validation
                    if (input.is('select[id*="company_address"], select[name*="company_address"]')) {
                        hideFieldError(input);
                        if (value === '' || value === null) {
                            showFieldError(input, 'Please select a region');
                        }
                    }
                    
                    checkFormValidityAndToggleButton(form);
                });
                
                // Also check after blur events
                form.on('blur', 'input, select, textarea', function() {
                    setTimeout(function() {
                        checkFormValidityAndToggleButton(form);
                    }, 100);
                });
                
                // Special listener for terms and conditions checkbox
                form.on('change', 'input[id*="jobhunt_check_terms"]', function() {
                    console.log('Terms checkbox changed:', $(this).is(':checked'));
                    checkFormValidityAndToggleButton(form);
                });
            });
            
            // Re-check forms when tabs are switched
            $('a[data-toggle="tab"]').on('shown.bs.tab', function(e) {
                setTimeout(function() {
                    $('form[id*="wp_signup_form"]').each(function() {
                        var form = $(this);
                        makeFieldsRequired(form);
                        checkFormValidityAndToggleButton(form);
                    });
                }, 500);
            });
            
        }, 1000); // Wait 1 second for dynamic content
        
        // ========================================
        // ENSURE PROFILE UPDATE BUTTONS ARE NEVER AFFECTED
        // ========================================
        
        // Remove any accidental disabled states from profile update buttons
        setTimeout(function() {
            $('.cs-section-update').each(function() {
                $(this).prop('disabled', false);
                $(this).removeClass('submit-disabled');
            });
        }, 1500);
        
        
    });
    </script>
    <?php
}


// Hook to modify form field labels and placeholders
add_filter('jobhunt_cand_reg_form', 'jobcareer_child_modify_candidate_registration_form', 10, 2);
function jobcareer_child_modify_candidate_registration_form($data, $rand_id) {
    // This filter allows us to modify the candidate registration form
    return $data;
}

// REMOVED - VALIDATION NOW HANDLED IN ORIGINAL JOBHUNT FILES

// Let the original registration function work without interference

// AJAX handler for checking duplicate email
add_action('wp_ajax_jobcareer_check_duplicate_email', 'jobcareer_child_check_duplicate_email');
add_action('wp_ajax_nopriv_jobcareer_check_duplicate_email', 'jobcareer_child_check_duplicate_email');

// Function to check duplicate email
function jobcareer_child_check_duplicate_email() {
    // Verify nonce
    if (!wp_verify_nonce($_POST['nonce'], 'email_check_nonce')) {
        wp_die('Security check failed');
    }
    
    $email = sanitize_email($_POST['email']);
    
    if (empty($email) || !is_email($email)) {
        wp_send_json_error('Invalid email format');
    }
    
    // Check if email already exists
    if (email_exists($email)) {
        wp_send_json_error('Email already exists');
    }
    
    wp_send_json_success('Email is available');
}

// Removed custom email verification - using JobHunt's system instead

// AJAX handler for auto-login after email verification
// ENABLED - Auto-login for candidates after email verification
add_action('wp_ajax_jobcareer_auto_login', 'jobcareer_child_auto_login');
add_action('wp_ajax_nopriv_jobcareer_auto_login', 'jobcareer_child_auto_login');

// Removed interfering registration validation function - let original work

// REMOVED - DUPLICATE FUNCTION (already declared above)

// Function to handle email verification
// Removed custom verification function - using JobHunt's system

// Function to handle auto-login after email verification (ONLY for candidates)
function jobcareer_child_auto_login() {
    $token = isset($_GET['token']) ? sanitize_text_field($_GET['token']) : '';
    $user_id = isset($_GET['user']) ? intval($_GET['user']) : 0;
    $redirect_to = isset($_GET['redirect_to']) ? urldecode($_GET['redirect_to']) : '';
    
    if (empty($token) || empty($user_id)) {
        wp_die('Invalid auto-login link');
    }
    
    // Get user data
    $user = get_userdata($user_id);
    if (!$user) {
        wp_die('User not found');
    }
    
    // ONLY allow auto-login for candidates
    if (!in_array('cs_candidate', $user->roles)) {
        wp_die('Auto-login is only available for candidates.');
    }
    
    // Check token validity (expires in 5 minutes)
    $stored_token = get_user_meta($user_id, 'temp_login_token', true);
    $token_time = get_user_meta($user_id, 'temp_login_token_time', true);
    
    if ($stored_token !== $token || (time() - $token_time) > 300) {
        wp_die('Auto-login link has expired. Please login manually.');
    }
    
    // Check if email is verified
    $is_verified = get_user_meta($user_id, 'email_verified', true);
    if (!$is_verified) {
        wp_die('Email not verified yet.');
    }
    
    error_log('JobCareer Child: Auto-login attempt for candidate user ID: ' . $user_id);
    
    // Perform the login
    wp_set_current_user($user_id);
    wp_set_auth_cookie($user_id, true);
    
    // Fire login action
    do_action('wp_login', $user->user_login, $user);
    
    // Clean up the temporary token
    delete_user_meta($user_id, 'temp_login_token');
    delete_user_meta($user_id, 'temp_login_token_time');
    
    error_log('JobCareer Child: Candidate auto-logged in successfully with ID: ' . $user_id);
    
    // Redirect to dashboard
    if (empty($redirect_to)) {
        $redirect_to = home_url('/candidate-dashboard/');
    }
    
    wp_redirect($redirect_to);
    exit;
}

// Hook into JobHunt's email verification redirect to add auto-login for candidates
// and send contract PDF for employers

// STEP 1: Intercept the verification process BEFORE the plugin redirects
// ENABLED - Capture user ID for auto-login and contract sending
add_action('init', 'jobcareer_child_capture_verifying_user', 1);
function jobcareer_child_capture_verifying_user() {
    if (isset($_GET['email_verify']) && $_GET['email_verify'] != '') {
        $email_verify = $_GET['email_verify'];
        $email_verify = base64_decode(base64_decode($email_verify));
        $email_verify = json_decode($email_verify);
        $user_id = isset($email_verify->user_id) ? intval($email_verify->user_id) : 0;
        
        // Store the user_id in a transient (expires in 60 seconds)
        if ($user_id > 0) {
            set_transient('jobcareer_verifying_user_id', $user_id, 60);
            error_log('JobCareer Child: Captured verifying user ID: ' . $user_id);
        }
    }
}

// Custom function to handle Arabic username login
function jobcareer_child_custom_login_username($user, $username, $password) {
    // If login failed with original username, try to find user by display name
    if (!$user && !empty($username)) {
        // Search for user by display name (Arabic name)
        $users = get_users(array(
            'meta_key' => 'display_name',
            'meta_value' => $username,
            'number' => 1
        ));
        
        if (!empty($users)) {
            $found_user = $users[0];
            // Verify password
            if (wp_check_password($password, $found_user->user_pass, $found_user->ID)) {
                return $found_user;
            }
        }
        
        // Also try to find by first_name or last_name
        $users = get_users(array(
            'meta_key' => 'first_name',
            'meta_value' => $username,
            'number' => 1
        ));
        
        if (!empty($users)) {
            $found_user = $users[0];
            if (wp_check_password($password, $found_user->user_pass, $found_user->ID)) {
                return $found_user;
            }
        }
    }
    
    return $user;
}
add_filter('authenticate', 'jobcareer_child_custom_login_username', 20, 3);

// STEP 2: Handle email verification after the redirect
// ENABLED - Auto-login for candidates and contract sending for employers
add_action('init', 'jobcareer_child_handle_email_verification_redirect', 5);

function jobcareer_child_handle_email_verification_redirect() {
    // Check if this is JobHunt's email verification redirect
    if (isset($_GET['email_verified']) && $_GET['email_verified'] == 'confirm') {
        // Get the user_id from the transient we set earlier
        $user_id = get_transient('jobcareer_verifying_user_id');
        
        if ($user_id) {
            // Delete the transient
            delete_transient('jobcareer_verifying_user_id');
            
            // Get user data
            $user = get_userdata($user_id);
            
            error_log('JobCareer Child: Email verification detected for user ID: ' . $user_id);
            
            // Check user role and handle accordingly
            if (in_array('cs_candidate', $user->roles)) {
                // ===== CANDIDATE: Auto-login =====
                error_log('JobCareer Child: User is a candidate, proceeding with auto-login');
                
                // Generate a temporary login token for auto-login
                $login_token = wp_generate_password(32, false);
                update_user_meta($user_id, 'temp_login_token', $login_token);
                update_user_meta($user_id, 'temp_login_token_time', time());
                
                // Get dashboard URL
                global $cs_plugin_options;
                $cs_candidate_dashboard = isset($cs_plugin_options['cs_js_dashboard']) ? $cs_plugin_options['cs_js_dashboard'] : '';
                
                if (!empty($cs_candidate_dashboard)) {
                    $dashboard_url = get_permalink($cs_candidate_dashboard);
                } else {
                    $dashboard_url = home_url('/candidate-dashboard/');
                }
                
                // Create auto-login URL
                $auto_login_url = add_query_arg(array(
                    'action' => 'jobcareer_auto_login',
                    'token' => $login_token,
                    'user' => $user_id,
                    'redirect_to' => urlencode($dashboard_url)
                ), admin_url('admin-ajax.php'));
                
                error_log('JobCareer Child: Redirecting candidate to auto-login URL: ' . $auto_login_url);
                
                // Redirect to auto-login
                wp_redirect($auto_login_url);
                exit;
                
            } elseif (in_array('cs_employer', $user->roles)) {
                // ===== EMPLOYER: Send contract PDF and keep logged out =====
                error_log('JobCareer Child: User is an employer, sending contract PDF');
                
                // Skip if contract already sent
                if (get_user_meta($user_id, 'contract_pdf_sent', true)) {
                    error_log('JobCareer Child: Contract already sent to employer ID: ' . $user_id);
                    wp_redirect(home_url('/?employer_verified=true'));
                    exit;
                }
                
                // Generate and send contract PDF
                jobcareer_child_send_employer_contract($user_id);
                
                // Mark as sent
                update_user_meta($user_id, 'contract_pdf_sent', true);
                
                // Redirect to a success page (keep user logged out)
                wp_redirect(home_url('/?employer_verified=true&contract_sent=true'));
                exit;
            }
        } else {
            // Fallback: transient not found (expired or verification link clicked multiple times)
            error_log('JobCareer Child: Transient not found for email verification - link may have expired or been used already');
            wp_redirect(home_url('/?verification_error=expired'));
            exit;
        }
    }
}

// Function to generate and send employer contract PDF
function jobcareer_child_send_employer_contract($user_id) {
    // Get user data
    $user = get_userdata($user_id);
    if (!$user) {
        error_log('JobCareer Child: User not found for contract generation: ' . $user_id);
        return false;
    }
    
    // Gather user data
    $first_name = $user->first_name;
    $last_name = $user->last_name;
    $user_name = trim($first_name . ' ' . $last_name);
    // If user_name is empty, fall back to display_name
    if (empty($user_name)) {
        $user_name = $user->display_name;
    }
    $phone = get_user_meta($user_id, 'cs_phone_number', true);
    $user_email = $user->user_email;
    $company_name = get_user_meta($user_id, 'cs_organization_name', true);
    $company_number = get_user_meta($user_id, 'cs_company_number', true);
    $company_address = get_user_meta($user_id, 'cs_company_address', true);
    $today = date('Y-m-d');
    
    // Debug logging for contract data
    error_log("JobCareer Child: Contract data for user {$user_id}:");
    error_log("  - user_name: " . $user_name);
    error_log("  - company_name: " . $company_name);
    error_log("  - company_number: " . $company_number);
    error_log("  - company_address: " . $company_address);
    error_log("  - user_email: " . $user_email);
    
    // Load HTML template
    $template_path = get_stylesheet_directory() . '/libs/mpdf/templates/contract-template.html';
    if (!file_exists($template_path)) {
        error_log("JobCareer Child: Failed to find contract template at: " . $template_path);
        return false;
    }
    
    $html = file_get_contents($template_path);
    if (!$html) {
        error_log("JobCareer Child: Failed to load contract template.");
        return false;
    }
    
    // Replace placeholders
    $replacements = [
        '[user_name]' => $user_name,
        '[user_email]' => $user_email,
        '[phone]' => $phone,
        '[company_name]' => $company_name,
        '[company_number]' => $company_number,
        '[company_address]' => $company_address,
        '[today]' => $today,
    ];
    $html = str_replace(array_keys($replacements), array_values($replacements), $html);
    
    // Load mPDF library
    if (!class_exists('\\Mpdf\\Mpdf')) {
        $autoload_path = get_stylesheet_directory() . '/libs/mpdf/vendor/autoload.php';
        if (file_exists($autoload_path)) {
            require_once $autoload_path;
        } else {
            error_log("JobCareer Child: mPDF library not found at: " . $autoload_path);
            return false;
        }
    }
    
    try {
        // Generate PDF
        $mpdf = new \Mpdf\Mpdf([
            'mode' => 'utf-8',
            'format' => 'A4',
            'default_font' => 'amiri',
            'margin_top' => 40,
            'margin_bottom' => 40,
        ]);
        $mpdf->WriteHTML($html);
        
        // Save to file in uploads directory
        $upload_dir = wp_upload_dir();
        $pdf_filename = "Employer_Contract_{$user_id}.pdf";
        $pdf_path = $upload_dir['basedir'] . "/" . $pdf_filename;
        $mpdf->Output($pdf_path, \Mpdf\Output\Destination::FILE);
        
        error_log("JobCareer Child: Contract PDF generated at: " . $pdf_path);
        
        // Send email with contract attachment
        $subject = "Employer Contract - عقد صاحب العمل";
        $message = 
"Dear {$user_name},

Attached is your employer agreement.

Please follow these steps:
1. Read the contract carefully.
2. Sign it at the bottom.
3. Reply to this email with the signed contract attached to complete your account activation.

Regards,  
Qeyada Careers

---

عزيزي/عزيزتي {$user_name}،

مرفق مع هذه الرسالة عقد صاحب العمل الخاص بك.

يرجى اتباع الخطوات التالية:
1. قراءة العقد بعناية والتأكد من صحة البيانات المدخلة.
2. التوقيع عليه في الأسفل.
3. إعادة إرسال العقد الموقع بالرد على هذا البريد الإلكتروني لاستكمال تفعيل الحساب.

مع التحية،  
قيادة للتوظيف
";
        
        $headers = ['Content-Type: text/plain; charset=UTF-8'];
        $attachments = [$pdf_path];
        
        $mail_sent = wp_mail($user_email, $subject, $message, $headers, $attachments);
        
        if ($mail_sent) {
            error_log("JobCareer Child: Contract email sent successfully to: " . $user_email);
            return true;
        } else {
            error_log("JobCareer Child: Failed to send contract email to: " . $user_email);
            return false;
        }
        
    } catch (Exception $e) {
        error_log("JobCareer Child: Error generating or sending contract PDF: " . $e->getMessage());
        return false;
    }
}

// Hook to modify the username field output with your custom validation
add_action('wp_footer', 'jobcareer_child_modify_username_field');
function jobcareer_child_modify_username_field() {
    ?>
    <script type="text/javascript">
    jQuery(document).ready(function($) {
        // Function to modify form fields
        function modifyFormFields() {
            // Change username field placeholder and add YOUR validation
            $('input[id*="user_login"], input[name*="user_login"], input[placeholder*="Username"]').each(function() {
                var input = $(this);
                
                // Change placeholder to "Full Name"
                input.attr('placeholder', 'Full Name');
                
                // Add YOUR custom attributes
                input.attr('pattern', '[a-zA-Z\\s]{3,}');
                input.attr('title', 'Please enter at least 3 characters using only letters and spaces');
                
                // Add custom class
                input.addClass('full-name-field');
                
                console.log('Modified username field:', input.attr('id'));
            });
            
            // Modify phone number fields to include Saudi flag and country code
            $('input[id*="phone"], input[name*="phone"], input[placeholder*="Phone"]').each(function() {
                var input = $(this);
                
                // Skip if already modified
                if (input.hasClass('phone-input')) {
                    return;
                }
                
                var container = input.closest('.field-holder');
                
                // Create new phone field container
                var phoneContainer = $('<div class="phone-field-container"></div>');
                var countryCode = $('<div class="phone-country-code"><span class="flag"></span>+966</div>');
                
                // Wrap the input and add country code
                input.wrap(phoneContainer);
                input.before(countryCode);
                input.addClass('phone-input');
                
                // Update placeholder
                input.attr('placeholder', '500000000');
                input.attr('maxlength', '9');
                input.attr('pattern', '5[0-9]{8}');
                input.attr('title', 'Enter 9-digit Saudi phone number starting with 5');
                
                console.log('Modified phone field:', input.attr('id'));
            });
        }
        
        // Run immediately
        modifyFormFields();
        
        // Run again after a delay to catch dynamic content
        setTimeout(modifyFormFields, 1000);
        setTimeout(modifyFormFields, 2000);
        setTimeout(modifyFormFields, 3000);
        
        // Also run when forms are loaded dynamically
        $(document).on('DOMNodeInserted', function() {
            setTimeout(modifyFormFields, 100);
        });
    });
    </script>
    <?php
}

// register theme menu
function cs_jobcareer_child_my_menus() {
    register_nav_menus(
            array(
                'main-menu' => __('Main Menu', 'jobcareer'),
                'footer-menu' => __('Footer Menu', 'jobcareer'),
            )
    );
}
add_action('init', 'cs_jobcareer_child_my_menus');

if (!get_option('cs_jobcareer_child')) {
    update_option('cs_jobcareer_child', 'jobcareer_child_theme');
    $theme_mod_val = array();
    $term_exists = term_exists('main-menu', 'nav_menu');
    if (!$term_exists) {
        $wpdb->insert(
                $wpdb->terms, array(
            'name' => 'Main Menu',
            'slug' => 'main-menu',
            'term_group' => 0
                ), array(
            '%s', '%s', '%d'
                )
        );
        $insert_id = $wpdb->insert_id;
        $theme_mod_val['main-menu'] = $insert_id;
        $wpdb->insert(
                $wpdb->term_taxonomy, array(
            'term_id' => $insert_id,
            'taxonomy' => 'nav_menu',
            'description' => '',
            'parent' => 0,
            'count' => 0
                ), array(
            '%d', '%s', '%s', '%d', '%d'
                )
        );
    } else {
        $theme_mod_val['main-menu'] = $term_exists['term_id'];
    }
    $term_exists = term_exists('top-menu', 'nav_menu');
    if (!$term_exists) {
        $wpdb->insert(
                $wpdb->terms, array(
            'name' => 'Top Menu',
            'slug' => 'top-menu',
            'term_group' => 0
                ), array(
            '%s', '%s', '%d'
                )
        );
        $insert_id = $wpdb->insert_id;
        $theme_mod_val['top-menu'] = $insert_id;
        $wpdb->insert(
                $wpdb->term_taxonomy, array(
            'term_id' => $insert_id,
            'taxonomy' => 'nav_menu',
            'description' => '',
            'parent' => 0,
            'count' => 0
                ), array(
            '%d', '%s', '%s', '%d', '%d'
                )
        );
    } else {
        $theme_mod_val['top-menu'] = $term_exists['term_id'];
    }

    $term_exists = term_exists('footer-menu', 'nav_menu');
    if (!$term_exists) {
        $wpdb->insert(
                $wpdb->terms, array(
            'name' => 'Footer Menu',
            'slug' => 'footer-menu',
            'term_group' => 0
                ), array(
            '%s', '%s', '%d'
                )
        );
        $insert_id = $wpdb->insert_id;
        $theme_mod_val['footer-menu'] = $insert_id;
        $wpdb->insert(
                $wpdb->term_taxonomy, array(
            'term_id' => $insert_id,
            'taxonomy' => 'nav_menu',
            'description' => '',
            'parent' => 0,
            'count' => 0
                ), array(
            '%d', '%s', '%s', '%d', '%d'
                )
        );
    } else {
        $theme_mod_val['footer-menu'] = $term_exists['term_id'];
    }
    set_theme_mod('nav_menu_locations', $theme_mod_val);
}


// =======================================
// Hide specific fields from candidate profile dashboard
// =======================================

// Hide Minimum Salary field from candidate profile
add_filter('jobhunt_candidate_minimum_salary_field', 'jobcareer_child_hide_minimum_salary_field');
function jobcareer_child_hide_minimum_salary_field($display) {
    return 'no';
}

// Prevent Google autocomplete fields from rendering completely
add_filter('jobhunt_candidate_fields_frontend', 'jobcareer_child_prevent_autocomplete_fields', 10, 1);
function jobcareer_child_prevent_autocomplete_fields($content) {
    // Remove any autocomplete-related content
    $content = preg_replace('/<div[^>]*autocomplete[^>]*>.*?<\/div>/is', '', $content);
    return $content;
}

// Prevent Google autocomplete function from running
add_filter('jobhunt_frontend_location_fields', '__return_empty_string', 999);
add_filter('cs_get_google_autocomplete_fields', '__return_empty_string', 999);

// =======================================
// Move CV section from CV & Cover Letter to My Resume tab
// =======================================

// Hook to add CV section to My Resume tab
add_action('jobhunt_user_fields_frontend_dashboard', 'jobcareer_child_add_cv_to_resume_tab');
function jobcareer_child_add_cv_to_resume_tab($uid) {
    global $cs_form_fields2;
    
    $cs_candidate_cv = get_user_meta($uid, 'cs_candidate_cv', true);
    ?>
    <section class="cs-tabs cs-cv-upload" id="cv-upload">
        <div class="field-holder">
            <h4><i class="icon-file4"></i><?php esc_html_e('Your CV', 'jobhunt'); ?></h4>
            <form id="candidate_cv_resume" name="cs_candidate_cv_resume" enctype="multipart/form-data" method="POST">
                <div class="upload-btn-div">
                <div class="dragareamain" style="padding-bottom:0px;">
                    <script type="text/ecmascript">
                        jQuery(document).ready(function(){
                        jQuery('.cs-uploadimg').change( function(e) {
                        var img = URL.createObjectURL(e.target.files[0]);
                        jQuery('#cs_candidate_cv').attr('value', img);
                        });
                        });
                    </script>
                    <div class="fileUpload uplaod-btn btn csborder-color cs-color">
                        <span class="cs-color"><?php esc_html_e('Browse', 'jobhunt'); ?></span>
                        <label class="browse-icon">
                            <?php
                            $cs_opt_array = array(
                                'std' => esc_html__('Browse', 'jobhunt'),
                                'cust_id' => 'media_upload',
                                'cust_name' => 'media_upload',
                                'cust_type' => 'file',
                                'force_std' => true,
                                'extra_atr' => ' onchange="checkName(this, \'cs_candidate_cv\', \'button_action\')"',
                                'classes' => 'upload cs-uploadimg cs-color csborder-color',
                            );
                            $cs_form_fields2->cs_form_text_render($cs_opt_array);
                            ?>
                        </label>
                    </div>
                    <div id="selecteduser-cv">
                        <?php
                        if (isset($cs_candidate_cv) and $cs_candidate_cv <> '' && (!isset($cs_candidate_cv['error']))) {
                            $cs_opt_array = array(
                                'std' => $cs_candidate_cv,
                                'cust_id' => 'cs_candidate_cv',
                                'cust_name' => 'cs_candidate_cv',
                                'cust_type' => 'hidden',
                            );
                            $cs_form_fields2->cs_form_text_render($cs_opt_array);
                            ?>
                            <div class="alert alert-dismissible user-resume" id="cs_candidate_cv_box">
                                <div>
                                    <?php
                                    if (isset($cs_candidate_cv) && $cs_candidate_cv != '') {
                                        if (cs_check_coverletter_exist($cs_candidate_cv)) {
                                            $uploads = wp_upload_dir();
                                            echo '<a target="_blank" href="' . esc_url($cs_candidate_cv) . '">';
                                            $parts = preg_split('~_(?=[^_]*$)~', basename($cs_candidate_cv));
                                            echo esc_html($parts[0]);
                                            echo '</a>';
                                            ?>
                                            <div class="gal-edit-opts close"><a href="javascript:cs_del_cover_letter('cs_candidate_cv')" class="delete">
                                                    <span aria-hidden="true">×</span></a>
                                            </div>
                                            <?php
                                        } else {
                                            esc_html_e("File not Available", "jobhunt");
                                        }
                                    }
                                    ?>
                                </div>
                            </div>
                        <?php } ?>				
                    </div>
                </div>
                <span class="cs-status-msg-cv-upload"><?php esc_html_e('Suitable files are .doc,docx,rft,pdf & .pdf', 'jobhunt'); ?></span>              
                </div>
                
                <?php
                // Hidden fields for form submission
                $cs_opt_array = array(
                    'std' => 'update_cv_profile',
                    'cust_id' => 'user_profile',
                    'cust_name' => 'user_profile',
                    'cust_type' => 'hidden',
                );
                $cs_form_fields2->cs_form_text_render($cs_opt_array);
                $cs_opt_array = array(
                    'std' => $uid,
                    'cust_id' => 'cs_user',
                    'cust_name' => 'cs_user',
                    'cust_type' => 'hidden',
                );
                $cs_form_fields2->cs_form_text_render($cs_opt_array);
                ?>
                
                <div class="cs-update-btn">
                    <a href="javascript:void(0);" name="button_action" class="acc-submit cs-section-update cs-color csborder-color" onclick="javascript:ajax_candidate_cv_form_save('<?php echo esc_js(admin_url('admin-ajax.php')); ?>', '<?php echo esc_js(wp_jobhunt::plugin_url()); ?>', 'candidate_cv_resume', '<?php echo absint($uid); ?>')"><?php esc_html_e('Update', 'jobhunt'); ?></a>
                    <?php
                    $cs_opt_array = array(
                        'std' => 'ajax_candidate_cv_form_save',
                        'cust_id' => 'action',
                        'cust_name' => 'action',
                        'cust_type' => 'hidden',
                    );
                    $cs_form_fields2->cs_form_text_render($cs_opt_array);
                    $cs_opt_array = array(
                        'std' => $uid,
                        'cust_id' => 'cs_user',
                        'cust_name' => 'cs_user',
                        'cust_type' => 'hidden',
                    );
                    $cs_form_fields2->cs_form_text_render($cs_opt_array);
                    ?>
                </div>
            </form>
        </div>
    </section>
    <?php
}

// Remove specific tabs from candidate dashboard menu
add_action('wp_footer', 'jobcareer_child_hide_dashboard_tabs');
function jobcareer_child_hide_dashboard_tabs() {
    if (is_user_logged_in() && current_user_can('cs_candidate')) {
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            function hideDashboardTabs() {
                // Hide English tabs from the menu
                $('a:contains("Packages")').closest('li').hide();
                $('a:contains("CV & Cover Letter")').closest('li').hide();
                $('a:contains("CV & cover letter")').closest('li').hide();
                $('a:contains("Job Alerts")').closest('li').hide();
                $('a:contains("Shortlisted jobs")').closest('li').hide();
                
                // Hide Arabic tabs from the menu
                $('a:contains("الباقات")').closest('li').hide();
                $('a:contains("السيرة الذاتية ورسالة التغطية")').closest('li').hide();
                $('a:contains("تنبيهات الوظائف")').closest('li').hide();
                $('a:contains("الوظائف المحفوظة")').closest('li').hide();
                $('a:contains("القائمة المختصرة")').closest('li').hide();
                $('a:contains("الوظائف المختصرة")').closest('li').hide();
                
                // Also hide from dropdown menus - English
                $('.dropdown-menu a:contains("Packages")').closest('li').hide();
                $('.dropdown-menu a:contains("CV & Cover Letter")').closest('li').hide();
                $('.dropdown-menu a:contains("CV & cover letter")').closest('li').hide();
                $('.dropdown-menu a:contains("Job Alerts")').closest('li').hide();
                $('.dropdown-menu a:contains("Shortlisted jobs")').closest('li').hide();
                
                // Also hide from dropdown menus - Arabic
                $('.dropdown-menu a:contains("الباقات")').closest('li').hide();
                $('.dropdown-menu a:contains("السيرة الذاتية ورسالة التغطية")').closest('li').hide();
                $('.dropdown-menu a:contains("تنبيهات الوظائف")').closest('li').hide();
                $('.dropdown-menu a:contains("الوظائف المحفوظة")').closest('li').hide();
                $('.dropdown-menu a:contains("القائمة المختصرة")').closest('li').hide();
                $('.dropdown-menu a:contains("الوظائف المختصرة")').closest('li').hide();
                
                // Hide from any menu structure
                $('a[href*="packages"], a[href*="cv-cover"], a[href*="job-alerts"], a[href*="shortlisted"]').closest('li').hide();
            }
            
            // Run immediately
            hideDashboardTabs();
            
            // Run after AJAX loads
            setTimeout(hideDashboardTabs, 1000);
            setTimeout(hideDashboardTabs, 2000);
            setTimeout(hideDashboardTabs, 3000);
        });
        </script>
        <?php
    }
}

// Remove specific tabs from employer dashboard menu
add_action('wp_footer', 'jobcareer_child_hide_employer_dashboard_tabs');
function jobcareer_child_hide_employer_dashboard_tabs() {
    if (is_user_logged_in() && current_user_can('cs_employer')) {
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            function hideEmployerDashboardTabs() {
                // Hide English tabs from the menu
                $('a:contains("Transactions")').closest('li').hide();
                $('a:contains("Packages")').closest('li').hide();
                $('a:contains("Job Alerts")').closest('li').hide();
                
                // Hide Arabic tabs from the menu
                $('a:contains("المعاملات")').closest('li').hide();
                $('a:contains("الباقات")').closest('li').hide();
                $('a:contains("تنبيهات الوظائف")').closest('li').hide();
                
                // Also hide from dropdown menus - English
                $('.dropdown-menu a:contains("Transactions")').closest('li').hide();
                $('.dropdown-menu a:contains("Packages")').closest('li').hide();
                $('.dropdown-menu a:contains("Job Alerts")').closest('li').hide();
                
                // Also hide from dropdown menus - Arabic
                $('.dropdown-menu a:contains("المعاملات")').closest('li').hide();
                $('.dropdown-menu a:contains("الباقات")').closest('li').hide();
                $('.dropdown-menu a:contains("تنبيهات الوظائف")').closest('li').hide();
                
                // Hide from any menu structure using href patterns
                $('a[href*="transactions"], a[href*="packages"], a[href*="job-alerts"]').closest('li').hide();
                
                // Hide by ID if they have specific IDs
                $('#employer_left_transactions_link').hide();
                $('#employer_left_packages_link').hide();
            }
            
            // Run immediately
            hideEmployerDashboardTabs();
            
            // Run after AJAX loads
            setTimeout(hideEmployerDashboardTabs, 1000);
            setTimeout(hideEmployerDashboardTabs, 2000);
            setTimeout(hideEmployerDashboardTabs, 3000);
        });
        </script>
        <?php
    }
}

// Hide Portfolio section from My Resume tab
add_action('wp_footer', 'jobcareer_child_hide_portfolio_from_resume');
function jobcareer_child_hide_portfolio_from_resume() {
    if (is_user_logged_in() && current_user_can('cs_candidate')) {
        ?>
        <style type="text/css">
            /* Hide Portfolio section from My Resume tab */
            .cs-portfolio,
            #portfolio,
            .cs-tabs.cs-portfolio {
                display: none !important;
            }
        </style>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Hide Portfolio section
            $('.cs-portfolio').hide();
            $('#portfolio').hide();
            $('.cs-tabs.cs-portfolio').hide();
            
            // Also hide after AJAX loads
            setTimeout(function() {
                $('.cs-portfolio').hide();
                $('#portfolio').hide();
                $('.cs-tabs.cs-portfolio').hide();
            }, 1000);
        });
        </script>
        <?php
    }
}

// Hook to remove CV section from CV & Cover Letter tab
add_action('wp_footer', 'jobcareer_child_hide_cv_from_cover_letter');
function jobcareer_child_hide_cv_from_cover_letter() {
    if (is_user_logged_in() && current_user_can('cs_candidate')) {
        ?>
        <style type="text/css">
            /* Hide CV section from CV & Cover Letter tab */
            .cs-cover-letter .resume-upload,
            .cs-cover-letter .inner-title:contains("Your CV"),
            .cs-cover-letter .upload-btn-div:has(.cs-uploadimg),
            .cs-cover-letter #selecteduser-cv {
                display: none !important;
            }
        </style>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Hide CV section from CV & Cover Letter tab
            $('.cs-cover-letter .resume-upload').hide();
            $('.cs-cover-letter .inner-title:contains("Your CV")').hide();
            $('.cs-cover-letter .inner-title:contains("Your CV")').nextAll().hide();
            
            // Also hide after AJAX loads
            setTimeout(function() {
                $('.cs-cover-letter .resume-upload').hide();
                $('.cs-cover-letter .inner-title:contains("Your CV")').hide();
            }, 1000);
        });
        </script>
        <?php
    }
}

// Add CSS to hide the Description field and Complete Address field
add_action('wp_head', 'jobcareer_child_hide_candidate_profile_fields');
function jobcareer_child_hide_candidate_profile_fields() {
    // Only apply on candidate dashboard
    if (is_user_logged_in() && current_user_can('cs_candidate')) {
        ?>
        <style type="text/css">
            /* Hide Description/Content Editor field and its label - English */
            .cs-profile-info .input-info .row .col-lg-12:has(label:contains("Description")) {
                display: none !important;
            }
            
            /* Hide Description/Content Editor field and its label - Arabic */
            .cs-profile-info .input-info .row .col-lg-12:has(label:contains("الوصف")) {
                display: none !important;
            }
            
            /* Hide Complete Address field specifically - Google autocomplete - English */
            input#autocomplete_address,
            label[for="autocomplete_address"],
            label:contains("Complete Address"),
            .social-input:has(input#autocomplete_address),
            .cs-social-network .social-media-info .social-input:has(#autocomplete_address),
            .col-lg-12:has(label:contains("Complete Address")),
            .input-info label:contains("Complete Address"),
            .cs-social-network label:contains("Complete Address"),
            .social-media-info label:contains("Complete Address") {
                display: none !important;
            }
            
            /* Hide Complete Address field specifically - Google autocomplete - Arabic */
            label:contains("العنوان كامل"),
            .col-lg-12:has(label:contains("العنوان كامل")),
            .input-info label:contains("العنوان كامل"),
            .cs-social-network label:contains("العنوان كامل"),
            .social-media-info label:contains("العنوان كامل") {
                display: none !important;
            }
            
            /* Hide Website field - English */
            input#user_url,
            label:contains("Website"),
            .social-input:has(input#user_url) {
                display: none !important;
            }
            
            /* Hide Website field - Arabic */
            label:contains("الموقع"),
            .social-input:has(input#user_url) {
                display: none !important;
            }
            
            /* Hide Cover Image field */
            #cs_cover_candidate_img_box,
            .cs-cover-img,
            .cs-img-detail:has(#cs_cover_candidate_img_box),
            .upload-btn-div:has(input[name="cand_cover_media_upload"]) {
                display: none !important;
            }
        </style>
        <?php
    }
}

// Use JavaScript to ensure the fields are hidden (backup method)
add_action('wp_footer', 'jobcareer_child_hide_profile_fields_js');
function jobcareer_child_hide_profile_fields_js() {
    if (is_user_logged_in() && current_user_can('cs_candidate')) {
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Function to hide all target fields
            function hideFields() {
                // Hide Description/Content Editor and its entire row - English
                $('label:contains("Description")').closest('.col-lg-12').hide();
                $('#wp-cs_candidate_content_editor-wrap').closest('.col-lg-12').hide();
                
                // Hide Description/Content Editor and its entire row - Arabic
                $('label:contains("الوصف")').closest('.col-lg-12').hide();
                
                // Hide Complete Address field specifically - English
                $('#autocomplete_address').closest('.social-input').hide();
                $('label[for="autocomplete_address"]').hide();
                $('label:contains("Complete Address")').hide();
                $('label:contains("Complete Address")').closest('.col-lg-12').hide();
                $('input#autocomplete_address').hide();
                
                // Hide Complete Address field specifically - Arabic
                $('label:contains("العنوان كامل")').hide();
                $('label:contains("العنوان كامل")').closest('.col-lg-12').hide();
                $('input[placeholder="العنوان كامل"]').hide();
                
                // Hide Website field - English
                $('input#user_url').closest('.social-input').hide();
                $('label:contains("Website")').closest('.social-input').hide();
                $('input#user_url').hide();
                
                // Hide Website field - Arabic
                $('label:contains("الموقع")').closest('.social-input').hide();
                $('input[placeholder="الموقع"]').hide();
                
                // Hide Cover Image field
                $('#cs_cover_candidate_img_box').closest('.cs-cover-img').hide();
                $('.cs-cover-img').hide();
                $('input[name="cand_cover_media_upload"]').closest('.upload-btn-div').hide();
                $('#cs_cover_candidate_img_box').hide();
            }
            
            // Hide fields immediately
            hideFields();
            
            // Also check for the fields after AJAX loads
            setTimeout(hideFields, 1000);
            setTimeout(hideFields, 2000);
            setTimeout(hideFields, 3000);
        });
        </script>
        <?php
    }
}

// ========================================
// CHANGE JOIN US BUTTON TO LINK TO CANDIDATE DASHBOARD
// ========================================
add_action('wp_footer', 'change_join_us_button_to_link', 999);
function change_join_us_button_to_link() {
    ?>
    <script type="text/javascript">
    jQuery(document).ready(function($) {
        // Change the Join Us button to link to candidate dashboard instead of opening modal
        $('a.join-us[data-target="#join-us"]').each(function() {
            $(this).attr('href', 'https://careers.qeyadasa.com/candidate-dashboard/');
            $(this).removeAttr('data-target');
            $(this).removeAttr('data-toggle');
        });
    });
    </script>
    <?php
}

// ========================================
// DEBUG FUNCTION TO CHECK EMPLOYER CONTRACT DATA
// ========================================
// Add this URL parameter to test: ?debug_contract_data=USER_ID (admin only)
add_action('init', 'debug_employer_contract_data');
function debug_employer_contract_data() {
    if (isset($_GET['debug_contract_data']) && current_user_can('manage_options')) {
        $user_id = intval($_GET['debug_contract_data']);
        $user = get_userdata($user_id);
        
        echo '<pre style="background: #f5f5f5; padding: 20px; margin: 20px; border: 1px solid #ccc;">';
        echo '<h2>Contract Data Debug for User ID: ' . $user_id . '</h2>';
        
        if (!$user) {
            echo 'ERROR: User not found!';
        } else {
            echo '<strong>WordPress User Fields:</strong>' . "\n";
            echo '  - user_login: ' . $user->user_login . "\n";
            echo '  - user_email: ' . $user->user_email . "\n";
            echo '  - display_name: ' . $user->display_name . "\n";
            echo '  - first_name: ' . $user->first_name . "\n";
            echo '  - last_name: ' . $user->last_name . "\n";
            echo '  - user_role: ' . implode(', ', $user->roles) . "\n\n";
            
            echo '<strong>User Meta Fields:</strong>' . "\n";
            echo '  - cs_organization_name: ' . get_user_meta($user_id, 'cs_organization_name', true) . "\n";
            echo '  - cs_company_number: ' . get_user_meta($user_id, 'cs_company_number', true) . "\n";
            echo '  - cs_company_address: ' . get_user_meta($user_id, 'cs_company_address', true) . "\n";
            echo '  - cs_phone_number: ' . get_user_meta($user_id, 'cs_phone_number', true) . "\n";
            echo '  - cr_number (old key): ' . get_user_meta($user_id, 'cr_number', true) . "\n\n";
            
            $first_name = $user->first_name;
            $last_name = $user->last_name;
            $user_name = trim($first_name . ' ' . $last_name);
            if (empty($user_name)) {
                $user_name = $user->display_name;
            }
            
            echo '<strong>Calculated Values for Contract:</strong>' . "\n";
            echo '  - user_name (for contract): ' . $user_name . "\n";
            echo '  - company_name: ' . get_user_meta($user_id, 'cs_organization_name', true) . "\n";
            echo '  - company_number: ' . get_user_meta($user_id, 'cs_company_number', true) . "\n";
            echo '  - company_address: ' . get_user_meta($user_id, 'cs_company_address', true) . "\n";
            
            echo "\n" . '<strong>Issues Found:</strong>' . "\n";
            if (empty($user_name) || $user_name == '') {
                echo '  ⚠️ WARNING: user_name is empty! User needs to set first_name and last_name.' . "\n";
            }
            if (empty(get_user_meta($user_id, 'cs_company_number', true))) {
                echo '  ⚠️ WARNING: company_number is empty! Check if cs_company_number meta exists.' . "\n";
            }
            if (empty(get_user_meta($user_id, 'cs_organization_name', true))) {
                echo '  ⚠️ WARNING: company_name is empty!' . "\n";
            }
            if (empty(get_user_meta($user_id, 'cs_company_address', true))) {
                echo '  ⚠️ WARNING: company_address is empty!' . "\n";
            }
        }
        
        echo '</pre>';
        die();
    }
}

// Replace specialisms dropdown with parent-only version in employer profile
add_filter('jobhunt_employer_profile_specialisms_dropdown', 'jobcareer_child_employer_specialisms_parent_only', 10, 4);
function jobcareer_child_employer_specialisms_parent_only($output, $name, $id, $user_id) {
    // Use the parent-only function instead
    return cs_get_specialisms_dropdown_parent_only($name, $id, $user_id, 'form-control chosen-select', true);
}

// Add CSS to hide Complete Address in employer profile
add_action('wp_head', 'jobcareer_child_hide_employer_complete_address');
function jobcareer_child_hide_employer_complete_address() {
    if (is_user_logged_in() && current_user_can('cs_employer')) {
        ?>
        <style type="text/css">
            /* Hide Complete Address field in employer profile - English */
            input#autocomplete_address,
            label:contains("Complete Address"),
            .col-lg-12:has(input#autocomplete_address),
            .col-lg-12:has(label:contains("Complete Address")) {
                display: none !important;
            }
            
            /* Hide Complete Address field in employer profile - Arabic */
            label:contains("العنوان كامل"),
            .col-lg-12:has(label:contains("العنوان كامل")) {
                display: none !important;
            }
        </style>
        <?php
    }
}

// Customize employer profile form - add custom fields and modify layout
add_action('wp_footer', 'jobcareer_child_customize_employer_profile_form');
function jobcareer_child_customize_employer_profile_form() {
    if (is_user_logged_in() && current_user_can('cs_employer')) {
        global $current_user;
        $user_id = $current_user->ID;
        
        // Get stored values
        $full_name = get_user_meta($user_id, 'cs_full_name', true);
        if (empty($full_name)) {
            // Fallback to username (user_login)
            $full_name = $current_user->user_login;
        }
        $job_title = get_user_meta($user_id, 'cs_job_title', true);
        
        // Get location values from user meta
        $city = get_user_meta($user_id, 'cs_post_loc_city', true);
        $country = get_user_meta($user_id, 'cs_post_loc_country', true);
        $username = $current_user->user_login;
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            function customizeEmployerProfileForm() {
                // Wait for the form to be loaded (AJAX loaded)
                if ($('#cs_employer_form').length > 0 && $('#cs_specialisms').length > 0) {
                    
                    // 1. Hide the Description field (textarea with editor)
                    var descriptionRow = $('#cs_comp_detail_editor').closest('.row');
                    if (descriptionRow.length > 0 && !descriptionRow.data('hidden')) {
                        descriptionRow.hide();
                        descriptionRow.data('hidden', true);
                    }
                    
                    // 1b. Hide Complete Address field (backup JavaScript method)
                    $('#autocomplete_address').closest('.col-lg-12').hide();
                    $('label:contains("Complete Address")').closest('.col-lg-12').hide();
                    $('label:contains("العنوان كامل")').closest('.col-lg-12').hide();
                    $('input#autocomplete_address').hide();
                    
                    // 1c. Ensure Country field shows the stored value
                    var countryField = $('#autocomplete_country');
                    if (countryField.length > 0) {
                        var storedCountry = '<?php echo esc_js($country); ?>';
                        if (storedCountry) {
                            if (countryField.val() === '' || countryField.val() === null) {
                                countryField.val(storedCountry);
                            }
                        }
                    }
                    
                    // Update the hidden country field
                    var hiddenCountryField = $('input[name="cs_post_loc_country"]');
                    if (hiddenCountryField.length > 0) {
                        var storedCountry = '<?php echo esc_js($country); ?>';
                        if (storedCountry) {
                            if (hiddenCountryField.val() === '' || hiddenCountryField.val() === null) {
                                hiddenCountryField.val(storedCountry);
                            }
                        }
                    }
                    
                    // 1d. Ensure City field shows the stored value
                    var cityField = $('#autocomplete_city');
                    if (cityField.length > 0) {
                        var storedCity = '<?php echo esc_js($city); ?>';
                        if (storedCity) {
                            if (cityField.val() === '' || cityField.val() === null) {
                                cityField.val(storedCity);
                            }
                        }
                    }
                    
                    // Also update the hidden city field
                    var hiddenCityField = $('input[name="cs_post_loc_city"]');
                    if (hiddenCityField.length > 0) {
                        var storedCity = '<?php echo esc_js($city); ?>';
                        if (storedCity) {
                            if (hiddenCityField.val() === '' || hiddenCityField.val() === null) {
                                hiddenCityField.val(storedCity);
                            }
                        }
                    }
                    
                    // 1e. Remove the separate Username display field since we're using it in Full Name
                    $('#employer_username_display').closest('.row').remove();
                    
                    // 2. Replace specialisms dropdown with parent-only version
                    var specialismsRow = $('#cs_specialisms').closest('.row');
                    if (specialismsRow.length > 0 && !specialismsRow.data('replaced')) {
                        specialismsRow.data('replaced', true);
                        
                        // Get the container
                        var specialismsContainer = $('#cs_specialisms').closest('.col-lg-6');
                        if (specialismsContainer.length > 0) {
                            // Replace with parent-only dropdown via AJAX
                            $.ajax({
                                url: '<?php echo admin_url('admin-ajax.php'); ?>',
                                type: 'POST',
                                data: {
                                    action: 'get_parent_specialisms_dropdown',
                                    user_id: <?php echo $user_id; ?>
                                },
                                success: function(response) {
                                    if (response.success) {
                                        // Replace the select holder content
                                        var selectHolder = specialismsContainer.find('.select-holder');
                                        if (selectHolder.length > 0) {
                                            selectHolder.html(response.data);
                                            // Reinitialize chosen
                                            setTimeout(function() {
                                                if (typeof $.fn.chosen !== 'undefined') {
                                                    $('#cs_specialisms').chosen({
                                                        width: '100%'
                                                    });
                                                }
                                            }, 100);
                                        }
                                    }
                                }
                            });
                        }
                    }
                    
                    // 3. Add Full Name and Job Title fields side by side (if not already added)
                    if ($('#cs_full_name').length === 0) {
                        var fullNameJobTitleHtml = '<div class="row" id="full_name_job_title_row">' +
                            '<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">' +
                            '<label>' + <?php echo json_encode(esc_html__('Full Name', 'jobhunt')); ?> + '</label>' +
                            '<input type="text" id="cs_full_name" name="cs_full_name" class="form-control" ' +
                            'placeholder="' + <?php echo json_encode(esc_html__('Full Name', 'jobhunt')); ?> + '" ' +
                            'value="<?php echo esc_js($full_name); ?>">' +
                            '</div>' +
                            '<div class="col-lg-6 col-md-6 col-sm-12 col-xs-12">' +
                            '<label>' + <?php echo json_encode(esc_html__('Job Title', 'jobhunt')); ?> + '</label>' +
                            '<div class="select-holder">' +
                            '<select id="cs_job_title" name="cs_job_title" class="form-control chosen-select-no-single">' +
                            '<option value="">' + <?php echo json_encode(esc_html__('Please Select', 'jobhunt')); ?> + '</option>' +
                            '<option value="c_level"' + (<?php echo json_encode($job_title); ?> === 'c_level' ? ' selected' : '') + '>' + 
                            <?php echo json_encode(esc_html__('C-Level Executive (e.g., CEO, CFO, CTO)', 'jobhunt')); ?> + '</option>' +
                            '<option value="department_head"' + (<?php echo json_encode($job_title); ?> === 'department_head' ? ' selected' : '') + '>' + 
                            <?php echo json_encode(esc_html__('Department Head/Manager', 'jobhunt')); ?> + '</option>' +
                            '<option value="hr_talent"' + (<?php echo json_encode($job_title); ?> === 'hr_talent' ? ' selected' : '') + '>' + 
                            <?php echo json_encode(esc_html__('HR/Talent Acquisition', 'jobhunt')); ?> + '</option>' +
                            '<option value="hiring_manager"' + (<?php echo json_encode($job_title); ?> === 'hiring_manager' ? ' selected' : '') + '>' + 
                            <?php echo json_encode(esc_html__('Hiring Manager', 'jobhunt')); ?> + '</option>' +
                            '<option value="recruiter"' + (<?php echo json_encode($job_title); ?> === 'recruiter' ? ' selected' : '') + '>' + 
                            <?php echo json_encode(esc_html__('Recruiter', 'jobhunt')); ?> + '</option>' +
                            '<option value="other"' + (<?php echo json_encode($job_title); ?> === 'other' ? ' selected' : '') + '>' + 
                            <?php echo json_encode(esc_html__('Other', 'jobhunt')); ?> + '</option>' +
                            '</select>' +
                            '</div>' +
                            '</div>' +
                            '</div>';
                        
                        // Insert after company name field
                        var companyNameRow = $('input[name="display_name"]').closest('.row');
                        if (companyNameRow.length > 0) {
                            companyNameRow.after(fullNameJobTitleHtml);
                            
                            // Re-initialize chosen select for the job title dropdown
                            setTimeout(function() {
                                if (typeof $.fn.chosen !== 'undefined') {
                                    $('#cs_job_title').chosen({
                                        disable_search: true,
                                        width: '100%'
                                    });
                                }
                            }, 100);
                        }
                    }
                }
            }
            
            // Run immediately
            customizeEmployerProfileForm();
            
            // Run after AJAX loads
            $(document).ajaxComplete(function() {
                setTimeout(customizeEmployerProfileForm, 500);
            });
            
            // Also run when profile tab is clicked
            setTimeout(customizeEmployerProfileForm, 1000);
            setTimeout(customizeEmployerProfileForm, 2000);
            setTimeout(customizeEmployerProfileForm, 3000);
        });
        </script>
        <?php
    }
}

// AJAX handler to get parent-only specialisms dropdown
add_action('wp_ajax_get_parent_specialisms_dropdown', 'jobcareer_child_ajax_parent_specialisms');
function jobcareer_child_ajax_parent_specialisms() {
    $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
    
    if ($user_id) {
        $dropdown = cs_get_specialisms_dropdown_parent_only('cs_specialisms', 'cs_specialisms', $user_id, 'form-control chosen-select', true);
        wp_send_json_success($dropdown);
    } else {
        wp_send_json_error();
    }
}

// Save custom employer profile fields
add_action('wp_ajax_cs_employer_profile', 'jobcareer_child_save_employer_custom_fields', 1);
function jobcareer_child_save_employer_custom_fields() {
    if (isset($_POST['cs_full_name'])) {
        $user_id = get_current_user_id();
        if ($user_id) {
            $full_name = sanitize_text_field($_POST['cs_full_name']);
            update_user_meta($user_id, 'cs_full_name', $full_name);
            
            // Also update first_name and last_name if possible
            $name_parts = explode(' ', $full_name, 2);
            if (count($name_parts) >= 1) {
                update_user_meta($user_id, 'first_name', $name_parts[0]);
                if (count($name_parts) >= 2) {
                    update_user_meta($user_id, 'last_name', $name_parts[1]);
                }
            }
        }
    }
    
    if (isset($_POST['cs_job_title'])) {
        $user_id = get_current_user_id();
        if ($user_id) {
            $job_title = sanitize_text_field($_POST['cs_job_title']);
            update_user_meta($user_id, 'cs_job_title', $job_title);
        }
    }
}

// Add Full Name and Job Title fields to employer registration form
add_action('wp_footer', 'jobcareer_child_add_employer_registration_fields');
function jobcareer_child_add_employer_registration_fields() {
    ?>
    <script type="text/javascript">
    jQuery(document).ready(function($) {
        function addEmployerRegistrationFields() {
            // Find the employer registration form
            var organizationField = $('input[id^="organization_name"]');
            
            if (organizationField.length > 0 && $('#cs_employer_full_name').length === 0) {
                var randId = organizationField.attr('id').replace('organization_name', '');
                
                // Add Full Name field after username
                var fullNameHtml = '<div class="col-md-12 col-lg-12 col-sm-12 col-xs-12">' +
                    '<div class="field-holder">' +
                    '<i class="icon-user9"></i>' +
                    '<input type="text" id="cs_employer_full_name' + randId + '" name="cs_employer_full_name' + randId + '" ' +
                    'class="form-control" placeholder="<?php echo esc_js(esc_html__('Full Name', 'jobhunt')); ?>" size="20">' +
                    '</div>' +
                    '</div>';
                
                // Insert after username field
                var usernameField = $('input[id="user_login_' + randId + '"]');
                if (usernameField.length > 0) {
                    usernameField.closest('.col-md-12').after(fullNameHtml);
                }
                
                // Add Job Title field after organization name
                var jobTitleHtml = '<div class="col-md-12 col-lg-12 col-sm-12 col-xs-12">' +
                    '<div class="select-holder">' +
                    '<select id="cs_employer_job_title' + randId + '" name="cs_employer_job_title' + randId + '" ' +
                    'class="form-control chosen-select" data-placeholder="<?php echo esc_js(esc_html__('Job Title', 'jobhunt')); ?>">' +
                    '<option value=""><?php echo esc_js(esc_html__('Job Title', 'jobhunt')); ?></option>' +
                    '<option value="c_level"><?php echo esc_js(esc_html__('C-Level Executive (e.g., CEO, CFO, CTO)', 'jobhunt')); ?></option>' +
                    '<option value="department_head"><?php echo esc_js(esc_html__('Department Head/Manager', 'jobhunt')); ?></option>' +
                    '<option value="hr_talent"><?php echo esc_js(esc_html__('HR/Talent Acquisition', 'jobhunt')); ?></option>' +
                    '<option value="hiring_manager"><?php echo esc_js(esc_html__('Hiring Manager', 'jobhunt')); ?></option>' +
                    '<option value="recruiter"><?php echo esc_js(esc_html__('Recruiter', 'jobhunt')); ?></option>' +
                    '<option value="other"><?php echo esc_js(esc_html__('Other', 'jobhunt')); ?></option>' +
                    '</select>' +
                    '</div>' +
                    '</div>';
                
                organizationField.closest('.col-md-12').after(jobTitleHtml);
                
                // Initialize chosen for job title dropdown
                setTimeout(function() {
                    if (typeof $.fn.chosen !== 'undefined') {
                        $('#cs_employer_job_title' + randId).chosen({
                            disable_search: true,
                            width: '100%'
                        });
                    }
                }, 500);
            }
        }
        
        // Run on page load
        setTimeout(addEmployerRegistrationFields, 500);
        setTimeout(addEmployerRegistrationFields, 1500);
        
        // Also run when modal opens or form is shown
        $(document).on('shown.bs.modal', addEmployerRegistrationFields);
        $(document).on('click', '.employer-tab', function() {
            setTimeout(addEmployerRegistrationFields, 500);
        });
    });
    </script>
    <?php
}

// Save employer registration custom fields - including location
add_action('jobhunt_after_registration', 'jobcareer_child_save_employer_registration_fields', 10, 2);
function jobcareer_child_save_employer_registration_fields($user_id, $post_data) {
    // Check if this is an employer registration
    $user = get_userdata($user_id);
    if ($user && in_array('cs_employer', $user->roles)) {
        error_log("JobCareer Child: Saving employer registration fields for user ID: $user_id");
        
        // Save full name
        foreach ($post_data as $key => $value) {
            if (strpos($key, 'cs_employer_full_name') !== false) {
                $full_name = sanitize_text_field($value);
                update_user_meta($user_id, 'cs_full_name', $full_name);
                error_log("JobCareer Child: Saved full name: $full_name");
                
                // Also update first_name and last_name
                $name_parts = explode(' ', $full_name, 2);
                if (count($name_parts) >= 1) {
                    update_user_meta($user_id, 'first_name', $name_parts[0]);
                    if (count($name_parts) >= 2) {
                        update_user_meta($user_id, 'last_name', $name_parts[1]);
                    }
                }
                break;
            }
        }
        
        // Save job title
        foreach ($post_data as $key => $value) {
            if (strpos($key, 'cs_employer_job_title') !== false) {
                $job_title = sanitize_text_field($value);
                update_user_meta($user_id, 'cs_job_title', $job_title);
                error_log("JobCareer Child: Saved job title: $job_title");
                break;
            }
        }
        
        // Save location fields (city, country, address)
        if (isset($post_data['cs_post_loc_city']) && !empty($post_data['cs_post_loc_city'])) {
            $city = sanitize_text_field($post_data['cs_post_loc_city']);
            update_user_meta($user_id, 'cs_post_loc_city', $city);
            error_log("JobCareer Child: Saved city: $city");
        }
        
        if (isset($post_data['cs_post_loc_country']) && !empty($post_data['cs_post_loc_country'])) {
            $country = sanitize_text_field($post_data['cs_post_loc_country']);
            update_user_meta($user_id, 'cs_post_loc_country', $country);
            error_log("JobCareer Child: Saved country: $country");
        }
        
        if (isset($post_data['cs_post_comp_address']) && !empty($post_data['cs_post_comp_address'])) {
            $address = sanitize_text_field($post_data['cs_post_comp_address']);
            update_user_meta($user_id, 'cs_post_comp_address', $address);
            error_log("JobCareer Child: Saved address: $address");
        }
        
        error_log("JobCareer Child: Employer registration fields saved successfully");
    }
}

/**
 * Restrict access to Candidates List page
 * Only allow employers, Recruitment Managers, and administrators
 */
add_action('template_redirect', 'restrict_candidates_list_page_access');

function restrict_candidates_list_page_access() {
    // Check if we're on the candidates list page
    if (is_page('candidates-list-with-filterable')) {
        
        // If user is not logged in, redirect to login page
        if (!is_user_logged_in()) {
            $login_url = wp_login_url(get_permalink());
            wp_redirect($login_url);
            exit;
        }
        
        // Get current user
        $current_user = wp_get_current_user();
        $user_roles = $current_user->roles;
        
        // Define allowed roles
        $allowed_roles = array('cs_employer', 'administrator', 'recruitment_manager');
        
        // Check if user has any of the allowed roles
        $has_access = false;
        foreach ($user_roles as $role) {
            if (in_array($role, $allowed_roles)) {
                $has_access = true;
                break;
            }
        }
        
        // If user doesn't have access, redirect to home page with error message
        if (!$has_access) {
            wp_redirect(home_url('/?access_denied=1'));
            exit;
        }
    }
}

/**
 * Remove contact form from candidate profile pages
 */
add_filter('jobhunt_candidate_contact_details_for_employers', 'remove_candidate_contact_form');

function remove_candidate_contact_form($display) {
    // Return 'no' to hide the contact form
    return 'no';
}

/**
 * Show only "View Profile" in the Actions dropdown for shortlisted resumes
 * Remove Cover Letter, Download CV, LinkedIn Profile, and Send Message options
 */
add_filter('jobhunt_candidate_action_check', 'show_only_view_profile_action', 10, 2);

function show_only_view_profile_action($action_flag, $candidate_id) {
    // Return false to hide all actions except "View Profile"
    // View Profile is displayed outside the action_flag check, so it will remain visible
    return false;
}

/**
 * Add custom CSS to hide unwanted action options in employer dashboard
 */
add_action('wp_head', 'hide_employer_dashboard_action_options');

function hide_employer_dashboard_action_options() {
    // Only add CSS on employer dashboard pages
    if (is_user_logged_in() && current_user_can('cs_employer')) {
        ?>
        <style type="text/css">
            /* Hide all action options except View Profile in employer dashboard */
            .cs-dash-resumes-tabs .cs-downlod-sec ul li a[data-target*="cover_letter_light"],
            .cs-dash-resumes-tabs .cs-downlod-sec ul li a[href*="linkedin.com"],
            .cs-dash-resumes-tabs .cs-downlod-sec ul li a[data-target*="cs-msgbox-"],
            .cs-dash-resumes-tabs .cs-downlod-sec ul li a:not([href*="author"]):not([href*="candidate"]) {
                display: none !important;
            }
            
            /* Hide list items that contain Cover Letter, Download CV, LinkedIn, Send Message */
            .cs-dash-resumes-tabs .cs-downlod-sec ul li:has(a[data-target*="cover_letter_light"]),
            .cs-dash-resumes-tabs .cs-downlod-sec ul li:has(a[href*="linkedin.com"]),
            .cs-dash-resumes-tabs .cs-downlod-sec ul li:has(a[data-target*="cs-msgbox-"]) {
                display: none !important;
            }
            
            /* Alternative approach: Show only View Profile links */
            .cs-dash-resumes-tabs .cs-downlod-sec ul li {
                display: none !important;
            }
            
            .cs-dash-resumes-tabs .cs-downlod-sec ul li:has(a[href*="/author/"]),
            .cs-dash-resumes-tabs .cs-downlod-sec ul li:has(a[href*="/candidate/"]) {
                display: list-item !important;
            }
        </style>
        <?php
    }
}
