<?php

namespace App\Http\Controllers\Api\Web;

use App\Http\Controllers\Controller;
use App\Mixins\Certificate\MakeCertificate;
use App\Models\Certificate;
use App\Models\CertificateTemplate;
use App\Models\Quiz;
use App\Models\QuizzesResult;
use App\Models\Webinar;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Storage;

class CertificatesController extends Controller
{
    /**
     * Get certificate image for mobile app
     * Returns saved PNG image file or generates new one if not exists
     */
    public function generateCertificateImage(Request $request)
    {
        try {
            $certificateId = $request->input('certificate_id');
            $type = $request->input('type', 'quiz'); // quiz, course, bundle
            $userId = $request->input('user_id');

            if (!$certificateId || !$userId) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate ID and User ID are required'
                ], 400);
            }

            // Get certificate based on type
            $certificate = null;
            
            switch ($type) {
                case 'quiz':
                    $certificate = $this->getQuizCertificate($certificateId, $userId);
                    break;
                    
                case 'course':
                    $certificate = $this->getCourseCertificate($certificateId, $userId);
                    break;
                    
                case 'bundle':
                    $certificate = $this->getBundleCertificate($certificateId, $userId);
                    break;
            }

            if (!$certificate) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate not found'
                ], 404);
            }

            // Check if image already exists
            $imagePath = public_path("store/certificates/certificate_{$certificate->id}.png");
            
            if (file_exists($imagePath)) {
                // Return existing image
                return response()->file($imagePath, [
                    'Content-Type' => 'image/png',
                    'Content-Disposition' => 'inline; filename="certificate.png"'
                ]);
            }

            // Generate new image using HTML2Canvas approach
            $html = $this->generateCertificateHtml($certificate, $type);
            $tempImagePath = $this->generateCertificateImageWithHtml2Canvas($html, $certificate->id);
            
            if (!$tempImagePath) {
                return response()->json([
                    'status' => false,
                    'message' => 'Failed to generate certificate image'
                ], 500);
            }

            // Move to permanent storage
            $permanentDir = dirname($imagePath);
            if (!file_exists($permanentDir)) {
                mkdir($permanentDir, 0755, true);
            }
            
            copy($tempImagePath, $imagePath);
            
            // Update certificate record
            $certificate->update([
                'image_path' => "store/certificates/certificate_{$certificate->id}.png"
            ]);
            
            // Clean up temporary file
            if (file_exists($tempImagePath)) {
                unlink($tempImagePath);
            }

            // Return the image file
            return response()->file($imagePath, [
                'Content-Type' => 'image/png',
                'Content-Disposition' => 'inline; filename="certificate.png"'
            ]);

        } catch (\Exception $e) {
            \Log::error('Certificate generation failed: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Certificate generation failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Enhanced certificate generation with better error handling and mobile optimization
     */
    public function generateCertificateImageEnhanced(Request $request)
    {
        try {
            $certificateId = $request->input('certificate_id');
            $type = $request->input('type', 'quiz');
            $userId = $request->input('user_id');
            $forceRegenerate = $request->input('force_regenerate', false);

            if (!$certificateId || !$userId) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate ID and User ID are required'
                ], 400);
            }

            \Log::info("Certificate generation request - ID: $certificateId, User: $userId, Type: $type");

            // Get certificate based on type
            $certificate = null;
            
            switch ($type) {
                case 'quiz':
                    $certificate = $this->getQuizCertificate($certificateId, $userId);
                    break;
                    
                case 'course':
                    $certificate = $this->getCourseCertificate($certificateId, $userId);
                    break;
                    
                case 'bundle':
                    $certificate = $this->getBundleCertificate($certificateId, $userId);
                    break;
                    
                case 'achievements':
                    // For achievements, try to get the certificate from the achievements table
                    $certificate = $this->getAchievementCertificate($certificateId, $userId);
                    break;
            }

            if (!$certificate) {
                \Log::error("Certificate not found - ID: $certificateId, User: $userId, Type: $type");
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate not found'
                ], 404);
            }

            $imagePath = public_path("store/certificates/certificate_{$certificate->id}.png");
            $imageUrl = url("store/certificates/certificate_{$certificate->id}.png");

            \Log::info("Certificate found - ID: {$certificate->id}, Image path: $imagePath");

            // Always generate the certificate image on-demand
            if ($forceRegenerate || !file_exists($imagePath) || filesize($imagePath) < 1000) {
                \Log::info("Generating certificate image for ID: {$certificate->id}, Type: {$type}");
                
                // Generate HTML content
                $html = $this->generateCertificateHtml($certificate, $type);
                
                if (!$html) {
                    \Log::error("Failed to generate certificate HTML");
                    return response()->json([
                        'status' => false,
                        'message' => 'Failed to generate certificate HTML'
                    ], 500);
                }

                \Log::info("HTML generated successfully, length: " . strlen($html));

                // Try HTML2Canvas first
                $tempImagePath = $this->generateCertificateImageWithHtml2Canvas($html, $certificate->id);
                
                if (!$tempImagePath) {
                    \Log::warning("HTML2Canvas failed, trying GD library fallback");
                    // Fallback to GD library
                    $tempImagePath = $this->generateCertificateImageFromHtml($html, $certificate->id);
                }
                
                if (!$tempImagePath) {
                    \Log::error("Failed to generate certificate image with both methods");
                    return response()->json([
                        'status' => false,
                        'message' => 'Failed to generate certificate image'
                    ], 500);
                }

                \Log::info("Image generated successfully at: $tempImagePath");

                // Ensure directory exists
                $permanentDir = dirname($imagePath);
                if (!file_exists($permanentDir)) {
                    mkdir($permanentDir, 0755, true);
                }
                
                // Move to permanent storage
                if (copy($tempImagePath, $imagePath)) {
                    // Update certificate record
                    $certificate->update([
                        'image_path' => "store/certificates/certificate_{$certificate->id}.png"
                    ]);
                    
                    \Log::info("Certificate image generated successfully: {$imagePath}");
                } else {
                    \Log::error("Failed to copy temporary image to permanent location");
                }
                
                // Clean up temporary file
                if (file_exists($tempImagePath)) {
                    unlink($tempImagePath);
                }
            } else {
                \Log::info("Certificate image already exists: $imagePath");
            }

            // Verify the image was created successfully
            if (!file_exists($imagePath) || filesize($imagePath) < 1000) {
                \Log::error("Certificate image verification failed - file doesn't exist or is too small");
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate image generation failed'
                ], 500);
            }

            // Return success response with image URL
            return response()->json([
                'status' => true,
                'message' => 'Certificate generated successfully',
                'data' => [
                    'image_url' => $imageUrl,
                    'certificate_id' => $certificate->id,
                    'type' => $type,
                    'exists' => file_exists($imagePath),
                    'file_size' => filesize($imagePath)
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('Enhanced certificate generation failed: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'status' => false,
                'message' => 'Certificate generation failed: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Generate certificate on-demand for mobile app (no authentication required)
     */
    public function generateCertificateOnDemand(Request $request)
    {
        try {
            $certificateId = $request->input('certificate_id');
            $userId = $request->input('user_id');
            $type = $request->input('type', 'quiz');

            if (!$certificateId || !$userId) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate ID and User ID are required'
                ], 400);
            }

            \Log::info("On-demand certificate generation - ID: $certificateId, User: $userId, Type: $type");

            // Get certificate based on type
            $certificate = null;
            
            switch ($type) {
                case 'quiz':
                    $certificate = $this->getQuizCertificate($certificateId, $userId);
                    break;
                    
                case 'course':
                    $certificate = $this->getCourseCertificate($certificateId, $userId);
                    break;
                    
                case 'bundle':
                    $certificate = $this->getBundleCertificate($certificateId, $userId);
                    break;
                    
                case 'achievements':
                    $certificate = $this->getAchievementCertificate($certificateId, $userId);
                    break;
            }

            if (!$certificate) {
                \Log::error("Certificate not found - ID: $certificateId, User: $userId, Type: $type");
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate not found'
                ], 404);
            }

            $imagePath = public_path("store/certificates/certificate_{$certificate->id}.png");
            $imageUrl = url("store/certificates/certificate_{$certificate->id}.png");

            \Log::info("Certificate found - ID: {$certificate->id}, Image path: $imagePath");

            // Always generate the certificate image
            \Log::info("Generating certificate image for ID: {$certificate->id}, Type: {$type}");
            
            // Generate HTML content
            $html = $this->generateCertificateHtml($certificate, $type);
            
            if (!$html) {
                \Log::error("Failed to generate certificate HTML");
                return response()->json([
                    'status' => false,
                    'message' => 'Failed to generate certificate HTML'
                ], 500);
            }

            \Log::info("HTML generated successfully, length: " . strlen($html));

            // Try HTML2Canvas first
            $tempImagePath = $this->generateCertificateImageWithHtml2Canvas($html, $certificate->id);
            
            if (!$tempImagePath) {
                \Log::warning("HTML2Canvas failed, trying GD library fallback");
                // Fallback to GD library
                $tempImagePath = $this->generateCertificateImageFromHtml($html, $certificate->id);
            }
            
            if (!$tempImagePath) {
                \Log::error("Failed to generate certificate image with both methods");
                return response()->json([
                    'status' => false,
                    'message' => 'Failed to generate certificate image'
                ], 500);
            }

            \Log::info("Image generated successfully at: $tempImagePath");

            // Ensure directory exists
            $permanentDir = dirname($imagePath);
            if (!file_exists($permanentDir)) {
                mkdir($permanentDir, 0755, true);
            }
            
            // Move to permanent storage
            if (copy($tempImagePath, $imagePath)) {
                // Update certificate record
                $certificate->update([
                    'image_path' => "store/certificates/certificate_{$certificate->id}.png"
                ]);
                
                \Log::info("Certificate image generated successfully: {$imagePath}");
            } else {
                \Log::error("Failed to copy temporary image to permanent location");
            }
            
            // Clean up temporary file
            if (file_exists($tempImagePath)) {
                unlink($tempImagePath);
            }

            // Verify the image was created successfully
            if (!file_exists($imagePath) || filesize($imagePath) < 1000) {
                \Log::error("Certificate image verification failed - file doesn't exist or is too small");
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate image generation failed'
                ], 500);
            }

            // Return success response with image URL
            return response()->json([
                'status' => true,
                'message' => 'Certificate generated successfully',
                'data' => [
                    'image_url' => $imageUrl,
                    'certificate_id' => $certificate->id,
                    'type' => $type,
                    'exists' => file_exists($imagePath),
                    'file_size' => filesize($imagePath)
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('On-demand certificate generation failed: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'status' => false,
                'message' => 'Certificate generation failed: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Generate HTML for certificate based on type
     */
    private function generateCertificateHtml($certificate, $type)
    {
        switch ($type) {
            case 'quiz':
                return $this->generateQuizCertificateHtml($certificate);
            case 'course':
                return $this->generateCourseCertificateHtml($certificate);
            case 'bundle':
                return $this->generateBundleCertificateHtml($certificate);
            default:
                return $this->generateDefaultCertificateHtml($certificate);
        }
    }

    /**
     * Get quiz certificate data
     */
    private function getQuizCertificate($certificateId, $userId)
    {
        return Certificate::where('id', $certificateId)
            ->where('student_id', $userId)
            ->where('type', 'quiz')
            ->with(['quizResult.quiz.webinar.teacher'])
            ->first();
    }

    /**
     * Get course certificate data
     */
    private function getCourseCertificate($certificateId, $userId)
    {
        return Certificate::where('id', $certificateId)
            ->where('student_id', $userId)
            ->where('type', 'course')
            ->with(['webinar.teacher'])
            ->first();
    }

    /**
     * Get bundle certificate data
     */
    private function getBundleCertificate($certificateId, $userId)
    {
        return Certificate::where('id', $certificateId)
            ->where('student_id', $userId)
            ->where('type', 'bundle')
            ->with(['bundle.teacher'])
            ->first();
    }

    /**
     * Generate HTML for quiz certificate
     */
    private function generateQuizCertificateHtml($certificate)
    {
        $template = CertificateTemplate::where('status', 'publish')
            ->where('type', 'quiz')
            ->first();

        if (!$template) {
            return $this->generateDefaultCertificateHtml($certificate);
        }

        $user = $certificate->student;
        $quiz = $certificate->quizResult->quiz;
        $webinar = $quiz->webinar;

        $body = $template->body;
        $body = $this->replaceCertificatePlaceholders($body, $certificate, $user, $webinar);

        return $this->wrapCertificateHtml($body);
    }

    /**
     * Generate HTML for course certificate
     */
    private function generateCourseCertificateHtml($certificate)
    {
        $template = CertificateTemplate::where('status', 'publish')
            ->where('type', 'course')
            ->first();

        if (!$template) {
            return $this->generateDefaultCertificateHtml($certificate);
        }

        $user = $certificate->student;
        $webinar = $certificate->webinar;

        $body = $template->body;
        $body = $this->replaceCertificatePlaceholders($body, $certificate, $user, $webinar);

        return $this->wrapCertificateHtml($body);
    }

    /**
     * Generate HTML for bundle certificate
     */
    private function generateBundleCertificateHtml($certificate)
    {
        $template = CertificateTemplate::where('status', 'publish')
            ->where('type', 'bundle')
            ->first();

        if (!$template) {
            return $this->generateDefaultCertificateHtml($certificate);
        }

        $user = $certificate->student;
        $bundle = $certificate->bundle;

        $body = $template->body;
        $body = $this->replaceCertificatePlaceholders($body, $certificate, $user, null, $bundle);

        return $this->wrapCertificateHtml($body);
    }

    /**
     * Generate default certificate HTML if no template exists
     */
    private function generateDefaultCertificateHtml($certificate)
    {
        $user = $certificate->student;
        $courseTitle = '';
        
        if ($certificate->webinar) {
            $courseTitle = $certificate->webinar->title;
        } elseif ($certificate->bundle) {
            $courseTitle = $certificate->bundle->title;
        }

        $html = "
        <div style='text-align: center; padding: 50px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; min-height: 400px; display: flex; flex-direction: column; justify-content: center;'>
            <div style='background: white; color: #333; padding: 40px; border-radius: 15px; box-shadow: 0 10px 30px rgba(0,0,0,0.3);'>
                <h1 style='color: #667eea; font-size: 32px; margin-bottom: 30px; font-family: Arial, sans-serif;'>Certificate of Completion</h1>
                <p style='font-size: 18px; margin-bottom: 20px; color: #666;'>This is to certify that</p>
                <h2 style='color: #333; font-size: 28px; margin-bottom: 30px; font-family: Arial, sans-serif; font-weight: bold;'>{$user->full_name}</h2>
                <p style='font-size: 16px; margin-bottom: 20px; color: #666;'>has successfully completed the course</p>
                <h3 style='color: #667eea; font-size: 22px; margin-bottom: 30px; font-family: Arial, sans-serif;'>{$courseTitle}</h3>
                <div style='border-top: 2px solid #667eea; padding-top: 20px; margin-top: 30px;'>
                    <p style='font-size: 14px; color: #666; margin: 5px 0;'>Certificate ID: {$certificate->id}</p>
                    <p style='font-size: 14px; color: #666; margin: 5px 0;'>Issued on: " . date('F j, Y', strtotime($certificate->created_at)) . "</p>
                </div>
            </div>
        </div>";

        return $html;
    }

    /**
     * Replace placeholders in certificate template
     */
    private function replaceCertificatePlaceholders($body, $certificate, $user, $webinar = null, $bundle = null)
    {
        $platformName = getGeneralSettings("site_name");
        $courseTitle = '';

        if ($webinar) {
            $courseTitle = $webinar->title;
        } elseif ($bundle) {
            $courseTitle = $bundle->title;
        }

        $replacements = [
            '[student]' => $user->full_name,
            '[student_name]' => $user->full_name,
            '[platform_name]' => $platformName,
            '[course]' => $courseTitle,
            '[course_name]' => $courseTitle,
            '[grade]' => $certificate->user_grade ?? '',
            '[certificate_id]' => $certificate->id,
            '[date]' => dateTimeFormat($certificate->created_at, 'j M Y | H:i'),
            '[instructor_name]' => $webinar ? $webinar->teacher->full_name : ($bundle ? $bundle->teacher->full_name : ''),
            '[duration]' => $webinar ? $webinar->duration : ($bundle ? $bundle->duration : ''),
        ];

        foreach ($replacements as $placeholder => $value) {
            $body = str_replace($placeholder, $value, $body);
        }

        return $body;
    }

    /**
     * Wrap certificate HTML with proper styling
     */
    private function wrapCertificateHtml($body)
    {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <style>
                body {
                    margin: 0;
                    padding: 0;
                    font-family: Arial, sans-serif;
                    background: white;
                }
                .certificate-container {
                    width: 800px;
                    height: 600px;
                    margin: 0 auto;
                    background: white;
                    position: relative;
                }
            </style>
        </head>
        <body>
            <div class='certificate-container'>
                {$body}
            </div>
        </body>
        </html>";
    }

    /**
     * Generate certificate image from HTML using GD library
     */
    private function generateCertificateImageFromHtml($html, $certificateId)
    {
        try {
            // Create image using GD library
            $width = 800;
            $height = 600;
            $image = imagecreatetruecolor($width, $height);
            
            // Set colors
            $white = imagecolorallocate($image, 255, 255, 255);
            $black = imagecolorallocate($image, 0, 0, 0);
            $blue = imagecolorallocate($image, 102, 126, 234);
            $purple = imagecolorallocate($image, 118, 75, 162);
            
            // Fill background with gradient
            $this->createGradientBackground($image, $width, $height, $blue, $purple);
            
            // Extract text content from HTML
            $textContent = strip_tags($html);
            
            // Add certificate content
            $this->addCertificateContent($image, $textContent, $certificateId);
            
            // Save image
            $outputPath = storage_path('app/temp/mobile_certificate_' . $certificateId . '.png');
            $tempDir = dirname($outputPath);
            
            if (!file_exists($tempDir)) {
                mkdir($tempDir, 0755, true);
            }
            
            imagepng($image, $outputPath);
            imagedestroy($image);
            
            return file_exists($outputPath) ? $outputPath : false;
            
        } catch (\Exception $e) {
            \Log::error('GD certificate generation failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Create gradient background
     */
    private function createGradientBackground($image, $width, $height, $color1, $color2)
    {
        for ($y = 0; $y < $height; $y++) {
            $ratio = $y / $height;
            $r = (int)((1 - $ratio) * 102 + $ratio * 118);
            $g = (int)((1 - $ratio) * 126 + $ratio * 75);
            $b = (int)((1 - $ratio) * 234 + $ratio * 162);
            
            $color = imagecolorallocate($image, $r, $g, $b);
            imageline($image, 0, $y, $width, $y, $color);
        }
    }

    /**
     * Add certificate content to image
     */
    private function addCertificateContent($image, $textContent, $certificateId)
    {
        $white = imagecolorallocate($image, 255, 255, 255);
        $black = imagecolorallocate($image, 0, 0, 0);
        $blue = imagecolorallocate($image, 102, 126, 234);
        
        // Create white background rectangle for content
        imagefilledrectangle($image, 50, 50, 750, 550, $white);
        
        // Add title
        $title = "Certificate of Completion";
        $font = public_path('assets/default/fonts/Montserrat-Medium.ttf');
        
        if (file_exists($font)) {
            imagettftext($image, 28, 0, 200, 120, $blue, $font, $title);
            
            // Add certificate ID
            $certId = "Certificate ID: " . $certificateId;
            imagettftext($image, 16, 0, 200, 160, $black, $font, $certId);
            
            // Add date
            $date = "Issued: " . date('F j, Y');
            imagettftext($image, 14, 0, 200, 190, $black, $font, $date);
            
            // Add main content (truncated for GD limitations)
            $lines = explode("\n", wordwrap($textContent, 50));
            $y = 250;
            foreach (array_slice($lines, 0, 8) as $line) {
                imagettftext($image, 12, 0, 100, $y, $black, $font, trim($line));
                $y += 30;
            }
        } else {
            // Fallback to built-in fonts
            imagestring($image, 5, 200, 120, $title, $blue);
            imagestring($image, 3, 200, 160, "Certificate ID: " . $certificateId, $black);
            imagestring($image, 3, 200, 190, "Issued: " . date('F j, Y'), $black);
        }
    }

    /**
     * Generate certificate image using HTML2Canvas approach
     * This creates a temporary HTML file and uses Puppeteer or similar to render it
     */
    private function generateCertificateImageWithHtml2Canvas($html, $certificateId)
    {
        try {
            // Create a temporary HTML file with the certificate content
            $tempHtmlPath = storage_path('app/temp/certificate_' . $certificateId . '.html');
            $tempDir = dirname($tempHtmlPath);
            
            if (!file_exists($tempDir)) {
                mkdir($tempDir, 0755, true);
            }

            // Create the HTML file with HTML2Canvas
            $htmlContent = $this->createHtml2CanvasPage($html);
            file_put_contents($tempHtmlPath, $htmlContent);

            // Use Chrome headless to render the HTML to PNG
            $outputPath = storage_path('app/temp/certificate_' . $certificateId . '.png');
            
            // Check if Chrome/Chromium is available
            $chromePath = $this->findChromePath();
            
            if ($chromePath) {
                // Use Chrome headless
                $command = sprintf(
                    '%s --headless --disable-gpu --screenshot="%s" --window-size=800,600 "%s"',
                    $chromePath,
                    $outputPath,
                    $tempHtmlPath
                );
                
                exec($command, $output, $returnCode);
                
                if ($returnCode === 0 && file_exists($outputPath)) {
                    // Clean up HTML file
                    if (file_exists($tempHtmlPath)) {
                        unlink($tempHtmlPath);
                    }
                    return $outputPath;
                }
            }

            // Fallback to GD library if Chrome is not available
            return $this->generateCertificateImageFromHtml($html, $certificateId);
            
        } catch (\Exception $e) {
            \Log::error('HTML2Canvas certificate generation failed: ' . $e->getMessage());
            return $this->generateCertificateImageFromHtml($html, $certificateId);
        }
    }

    /**
     * Create HTML page with HTML2Canvas for certificate generation
     */
    private function createHtml2CanvasPage($certificateHtml)
    {
        return '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Certificate Generator</title>
    <style>
        body {
            margin: 0;
            padding: 0;
            background: white;
            font-family: Arial, sans-serif;
        }
        .certificate-container {
            width: 800px;
            height: 600px;
            margin: 0 auto;
            background: white;
            position: relative;
        }
    </style>
</head>
<body>
    <div class="certificate-container">
        ' . $certificateHtml . '
    </div>
    
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
    <script>
        // Wait for page to load
        window.onload = function() {
            setTimeout(function() {
                const element = document.querySelector(".certificate-container");
                
                html2canvas(element, {
                    allowTaint: true,
                    useCORS: true,
                    scale: 2,
                    width: 800,
                    height: 600,
                    backgroundColor: "#ffffff"
                }).then(function(canvas) {
                    // Convert canvas to data URL and send to server
                    const dataUrl = canvas.toDataURL("image/png");
                    
                    // Create a form to submit the image data
                    const form = document.createElement("form");
                    form.method = "POST";
                    form.action = "/api/development/certificates/save-temp-image";
                    
                    const input = document.createElement("input");
                    input.type = "hidden";
                    input.name = "image_data";
                    input.value = dataUrl;
                    
                    form.appendChild(input);
                    document.body.appendChild(form);
                    form.submit();
                });
            }, 1000);
        };
    </script>
</body>
</html>';
    }

    /**
     * Find Chrome/Chromium executable path
     */
    private function findChromePath()
    {
        $possiblePaths = [
            '/usr/bin/google-chrome',
            '/usr/bin/chromium-browser',
            '/usr/bin/chromium',
            '/Applications/Google Chrome.app/Contents/MacOS/Google Chrome',
            'C:\Program Files\Google\Chrome\Application\chrome.exe',
            'C:\Program Files (x86)\Google\Chrome\Application\chrome.exe'
        ];

        foreach ($possiblePaths as $path) {
            if (file_exists($path)) {
                return $path;
            }
        }

        return null;
    }

    /**
     * Get certificate HTML content for mobile WebView rendering
     */
    public function getCertificateHtml(Request $request)
    {
        try {
            $certificateId = $request->input('certificate_id');
            $type = $request->input('type', 'quiz');
            $userId = $request->input('user_id');

            if (!$certificateId || !$userId) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate ID and User ID are required'
                ], 400);
            }

            // Get certificate based on type
            $certificate = null;
            
            switch ($type) {
                case 'quiz':
                    $certificate = $this->getQuizCertificate($certificateId, $userId);
                    break;
                    
                case 'course':
                    $certificate = $this->getCourseCertificate($certificateId, $userId);
                    break;
                    
                case 'bundle':
                    $certificate = $this->getBundleCertificate($certificateId, $userId);
                    break;
            }

            if (!$certificate) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate not found'
                ], 404);
            }

            // Generate HTML content
            $html = $this->generateCertificateHtml($certificate, $type);
            
            // Create mobile-optimized HTML
            $mobileHtml = $this->createMobileCertificateHtml($html);

            return response()->json([
                'status' => true,
                'data' => [
                    'html' => $mobileHtml,
                    'certificate_id' => $certificate->id,
                    'type' => $type
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('Mobile certificate HTML generation failed: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Certificate HTML generation failed'
            ], 500);
        }
    }

    /**
     * Create mobile-optimized HTML for certificate display
     */
    private function createMobileCertificateHtml($certificateHtml)
    {
        return '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Certificate</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: Arial, sans-serif;
            background: #f5f5f5;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            padding: 20px;
        }
        
        .certificate-wrapper {
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            overflow: hidden;
            max-width: 100%;
            max-height: 100vh;
        }
        
        .certificate-container {
            width: 800px;
            height: 600px;
            position: relative;
            background: white;
            transform-origin: top left;
        }
        
        @media (max-width: 850px) {
            .certificate-container {
                transform: scale(0.8);
                transform-origin: top left;
            }
        }
        
        @media (max-width: 680px) {
            .certificate-container {
                transform: scale(0.6);
                transform-origin: top left;
            }
        }
        
        @media (max-width: 510px) {
            .certificate-container {
                transform: scale(0.5);
                transform-origin: top left;
            }
        }
        
        .certificate-container img {
            max-width: 100%;
            height: auto;
        }
        
        .certificate-container * {
            max-width: 100%;
        }
    </style>
</head>
<body>
    <div class="certificate-wrapper">
        <div class="certificate-container">
            ' . $certificateHtml . '
        </div>
    </div>
    
    <script>
        // Handle image loading for better mobile experience
        document.addEventListener("DOMContentLoaded", function() {
            const images = document.querySelectorAll("img");
            images.forEach(function(img) {
                img.addEventListener("load", function() {
                    // Image loaded successfully
                });
                img.addEventListener("error", function() {
                    // Handle image load error
                    this.style.display = "none";
                });
            });
        });
        
        // Prevent zoom on double tap (iOS)
        let lastTouchEnd = 0;
        document.addEventListener("touchend", function (event) {
            const now = (new Date()).getTime();
            if (now - lastTouchEnd <= 300) {
                event.preventDefault();
            }
            lastTouchEnd = now;
        }, false);
    </script>
</body>
</html>';
    }

    /**
     * Get certificate list for mobile app
     */
    public function getCertificates(Request $request)
    {
        $userId = $request->input('user_id');
        
        if (!$userId) {
            return response()->json([
                'status' => false,
                'message' => 'User ID is required'
            ], 400);
        }

        $certificates = Certificate::where('student_id', $userId)
            ->with(['webinar', 'bundle', 'quizResult.quiz.webinar'])
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($certificate) {
                $data = [
                    'id' => $certificate->id,
                    'type' => $certificate->type,
                    'created_at' => $certificate->created_at,
                    'image_url' => null
                ];

                // Add course/bundle title
                if ($certificate->webinar) {
                    $data['title'] = $certificate->webinar->title;
                } elseif ($certificate->bundle) {
                    $data['title'] = $certificate->bundle->title;
                } elseif ($certificate->quizResult && $certificate->quizResult->quiz->webinar) {
                    $data['title'] = $certificate->quizResult->quiz->webinar->title;
                } else {
                    $data['title'] = 'Unknown Course';
                }

                // Add image URL if exists
                if ($certificate->image_path) {
                    $data['image_url'] = asset($certificate->image_path);
                }

                return $data;
            });

        return response()->json([
            'status' => true,
            'data' => $certificates
        ]);
    }

    /**
     * Test certificate generation endpoint
     */
    public function testCertificateGeneration(Request $request)
    {
        try {
            $certificateId = $request->input('certificate_id');
            $userId = $request->input('user_id');
            $type = $request->input('type', 'quiz');

            if (!$certificateId || !$userId) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate ID and User ID are required'
                ], 400);
            }

            // Test HTML generation
            $certificate = $this->getQuizCertificate($certificateId, $userId);
            if (!$certificate) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate not found'
                ], 404);
            }

            $html = $this->generateCertificateHtml($certificate, $type);
            
            return response()->json([
                'status' => true,
                'message' => 'Certificate test successful',
                'data' => [
                    'certificate_id' => $certificate->id,
                    'type' => $type,
                    'html_length' => strlen($html),
                    'image_path' => public_path("store/certificates/certificate_{$certificate->id}.png"),
                    'image_exists' => file_exists(public_path("store/certificates/certificate_{$certificate->id}.png"))
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Test failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get achievement certificate
     */
    private function getAchievementCertificate($certificateId, $userId)
    {
        try {
            // Try to get from achievements table
            $achievement = \App\Models\Achievement::where('id', $certificateId)
                ->where('user_id', $userId)
                ->first();
                
            if ($achievement) {
                return $achievement;
            }
            
            // Fallback to certificate table
            return \App\Models\Certificate::where('id', $certificateId)
                ->where('user_id', $userId)
                ->first();
                
        } catch (\Exception $e) {
            \Log::error("Error getting achievement certificate: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Simple public certificate generation for mobile apps
     */
    public function generateCertificatePublic(Request $request)
    {
        try {
            $certificateId = $request->input('certificate_id');
            $userId = $request->input('user_id');
            
            if (!$certificateId || !$userId) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate ID and User ID are required'
                ], 400);
            }

            \Log::info("Public certificate generation - ID: $certificateId, User: $userId");

            // Check if certificate image already exists
            $imagePath = public_path("store/certificates/certificate_$certificateId.png");
            
            if (file_exists($imagePath) && filesize($imagePath) > 1000) {
                \Log::info("Certificate image already exists: $imagePath");
                return response()->json([
                    'status' => true,
                    'message' => 'Certificate already exists',
                    'image_url' => url("store/certificates/certificate_$certificateId.png")
                ]);
            }

            // Find the quiz result
            $quizResult = QuizzesResult::where('id', $certificateId)
                ->where('user_id', $userId)
                ->where('status', QuizzesResult::$passed)
                ->with(['quiz' => function ($query) {
                    $query->with(['webinar']);
                }])
                ->first();

            if (empty($quizResult)) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate not found or user not authorized'
                ], 404);
            }

            // Generate certificate using the existing system
            $makeCertificate = new MakeCertificate();
            $result = $makeCertificate->makeQuizCertificate($quizResult);

            if ($result) {
                \Log::info("Certificate generated successfully: $imagePath");
                return response()->json([
                    'status' => true,
                    'message' => 'Certificate generated successfully',
                    'image_url' => url("store/certificates/certificate_$certificateId.png")
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Failed to generate certificate'
                ], 500);
            }

        } catch (\Exception $e) {
            \Log::error("Certificate generation error: " . $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Certificate generation failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Enhanced certificate generation with offline support
     */
    public function generateCertificateEnhanced(Request $request)
    {
        try {
            $certificateId = $request->input('certificate_id');
            $userId = $request->input('user_id');
            $type = $request->input('type', 'quiz');
            $forceRegenerate = $request->input('force_regenerate', false);
            
            if (!$certificateId || !$userId) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate ID and User ID are required'
                ], 400);
            }

            \Log::info("Enhanced certificate generation - ID: $certificateId, User: $userId, Type: $type");

            // Find the certificate
            $certificate = Certificate::where('id', $certificateId)
                ->where('student_id', $userId)
                ->first();

            if (empty($certificate)) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate not found'
                ], 404);
            }

            // Check if certificate already exists and is valid
            if (!$forceRegenerate && $certificate->isGenerated()) {
                return response()->json([
                    'status' => true,
                    'message' => 'Certificate already exists',
                    'data' => [
                        'certificate_id' => $certificate->id,
                        'image_url' => $certificate->image_url,
                        'html_content' => $certificate->html_content,
                        'qr_code' => $certificate->qr_code,
                        'verification_url' => $certificate->verification_url,
                        'generated_at' => $certificate->generated_at,
                        'offline_data' => $this->getOfflineData($certificate)
                    ]
                ]);
            }

            // Generate certificate using rich template system
            $makeCertificate = new MakeCertificate();
            
            if ($type === 'quiz') {
                $quizResult = QuizzesResult::where('id', $certificateId)
                    ->where('user_id', $userId)
                    ->where('status', QuizzesResult::$passed)
                    ->with(['quiz' => function ($query) {
                        $query->with(['webinar']);
                    }])
                    ->first();

                if (empty($quizResult)) {
                    return response()->json([
                        'status' => false,
                        'message' => 'Quiz result not found or user not authorized'
                    ], 404);
                }

                $result = $makeCertificate->makeQuizCertificate($quizResult);
            } elseif ($type === 'course') {
                $result = $makeCertificate->makeCourseCertificate($certificate);
            } elseif ($type === 'bundle') {
                $result = $makeCertificate->makeBundleCertificate($certificate);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Invalid certificate type'
                ], 400);
            }

            if ($result) {
                // Update certificate status
                $certificate->update([
                    'status' => Certificate::STATUS_GENERATED,
                    'generated_at' => now(),
                    'image_path' => "store/certificates/certificate_{$certificate->id}.png"
                ]);

                return response()->json([
                    'status' => true,
                    'message' => 'Certificate generated successfully',
                    'data' => [
                        'certificate_id' => $certificate->id,
                        'image_url' => $certificate->image_url,
                        'html_content' => $certificate->html_content,
                        'qr_code' => $certificate->qr_code,
                        'verification_url' => $certificate->verification_url,
                        'generated_at' => $certificate->generated_at,
                        'offline_data' => $this->getOfflineData($certificate)
                    ]
                ]);
            } else {
                $certificate->update(['status' => Certificate::STATUS_FAILED]);
                
                return response()->json([
                    'status' => false,
                    'message' => 'Failed to generate certificate'
                ], 500);
            }

        } catch (\Exception $e) {
            \Log::error("Enhanced certificate generation error: " . $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Certificate generation failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get certificate with offline data
     */
    public function getCertificateWithOfflineData(Request $request)
    {
        try {
            $certificateId = $request->input('certificate_id');
            $userId = $request->input('user_id');
            
            if (!$certificateId || !$userId) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate ID and User ID are required'
                ], 400);
            }

            $certificate = Certificate::where('id', $certificateId)
                ->where('student_id', $userId)
                ->first();

            if (empty($certificate)) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certificate not found'
                ], 404);
            }

            return response()->json([
                'status' => true,
                'data' => [
                    'certificate_id' => $certificate->id,
                    'image_url' => $certificate->image_url,
                    'html_content' => $certificate->html_content,
                    'qr_code' => $certificate->qr_code,
                    'verification_url' => $certificate->verification_url,
                    'generated_at' => $certificate->generated_at,
                    'is_generated' => $certificate->isGenerated(),
                    'offline_data' => $this->getOfflineData($certificate)
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error("Get certificate error: " . $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Failed to get certificate: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get certificates list
     */
    public function getCertificatesList(Request $request)
    {
        try {
            $userId = $request->input('user_id');
            $type = $request->input('type');
            
            if (!$userId) {
                return response()->json([
                    'status' => false,
                    'message' => 'User ID is required'
                ], 400);
            }

            $query = Certificate::where('student_id', $userId);

            if ($type) {
                $query->where('type', $type);
            }

            $certificates = $query->with(['student', 'quiz', 'webinar', 'bundle'])
                ->orderBy('created_at', 'desc')
                ->get();

            $certificatesData = $certificates->map(function ($certificate) {
                return [
                    'id' => $certificate->id,
                    'student_id' => $certificate->student_id,
                    'quiz_id' => $certificate->quiz_id,
                    'webinar_id' => $certificate->webinar_id,
                    'bundle_id' => $certificate->bundle_id,
                    'type' => $certificate->type,
                    'status' => $certificate->status,
                    'user_grade' => $certificate->user_grade,
                    'image_path' => $certificate->image_path,
                    'html_content' => $certificate->html_content,
                    'qr_code' => $certificate->qr_code,
                    'verification_url' => $certificate->verification_url,
                    'created_at' => $certificate->created_at,
                    'generated_at' => $certificate->generated_at,
                    'metadata' => $certificate->metadata,
                    'offline_data' => $this->getOfflineData($certificate),
                ];
            });

            return response()->json([
                'status' => true,
                'data' => $certificatesData
            ]);

        } catch (\Exception $e) {
            \Log::error("Get certificates list error: " . $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Failed to get certificates: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get offline data for certificate
     */
    private function getOfflineData($certificate)
    {
        // Handle integer timestamp for created_at
        $issuedDate = '';
        if ($certificate->created_at) {
            if (is_numeric($certificate->created_at)) {
                $issuedDate = date('F j, Y', $certificate->created_at);
            } else {
                $issuedDate = $certificate->created_at->format('F j, Y');
            }
        }
        
        return [
            'certificate_id' => $certificate->id,
            'student_name' => $certificate->student->full_name ?? '',
            'course_title' => $this->getCourseTitle($certificate),
            'grade' => $certificate->user_grade ?? '',
            'issued_date' => $issuedDate,
            'verification_code' => $this->generateVerificationCode($certificate),
            'template_data' => $this->getTemplateData($certificate)
        ];
    }

    /**
     * Get course title for certificate
     */
    private function getCourseTitle($certificate)
    {
        if ($certificate->quiz && $certificate->quiz->webinar) {
            return $certificate->quiz->webinar->title;
        } elseif ($certificate->webinar) {
            return $certificate->webinar->title;
        } elseif ($certificate->bundle) {
            return $certificate->bundle->title;
        }
        return '';
    }

    /**
     * Generate verification code for certificate
     */
    private function generateVerificationCode($certificate)
    {
        return strtoupper(substr(md5($certificate->id . $certificate->student_id), 0, 8));
    }

    /**
     * Get template data for certificate
     */
    private function getTemplateData($certificate)
    {
        $template = $certificate->template;
        if (!$template) {
            $template = CertificateTemplate::where('status', 'publish')
                ->where('type', $certificate->type ?? 'quiz')
                ->first();
        }

        return [
            'template_id' => $template->id ?? null,
            'template_name' => $template->title ?? '',
            'template_type' => $template->type ?? 'quiz',
            'template_body' => $template->body ?? '',
            'template_elements' => $template->elements ?? []
        ];
    }
}
