<?php

namespace App\Http\Controllers\Api\Web;

use App\Http\Controllers\Api\Controller;
use App\Models\Certification;
use App\Models\User;
use Illuminate\Http\Request;

class CertificationsController extends Controller
{
    /**
     * Get certifications for the authenticated user
     */
    public function index(Request $request)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Match certifications where id_no OR govt_id matches the user's student_id
            $certifications = Certification::where(function($query) use ($user) {
                $query->where('id_no', $user->student_id)
                      ->orWhere('govt_id', $user->student_id);
            })
            ->orderBy('Id', 'desc')
            ->get();

            return response()->json([
                'status' => true,
                'message' => 'Certifications retrieved successfully',
                'data' => $certifications
            ]);

        } catch (\Exception $e) {
            \Log::error('Certifications API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error retrieving certifications',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get a specific certification
     */
    public function show($id)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Match certification where Id matches the requested id AND (id_no OR govt_id matches the user's student_id)
            $certification = Certification::where('Id', $id)
                ->where(function($query) use ($user) {
                    $query->where('id_no', $user->student_id)
                          ->orWhere('govt_id', $user->student_id);
                })
                ->first();

            if (!$certification) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certification not found'
                ], 404);
            }

            return response()->json([
                'status' => true,
                'message' => 'Certification retrieved successfully',
                'data' => $certification
            ]);

        } catch (\Exception $e) {
            \Log::error('Certification show API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error retrieving certification',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download a specific certification
     */
    public function download($id)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Match certification where Id matches the requested id AND (id_no OR govt_id matches the user's student_id)
            $certification = Certification::where('Id', $id)
                ->where(function($query) use ($user) {
                    $query->where('id_no', $user->student_id)
                          ->orWhere('govt_id', $user->student_id);
                })
                ->first();

            if (!$certification) {
                return response()->json([
                    'status' => false,
                    'message' => 'Certification not found'
                ], 404);
            }

            // For now, return the certification data as JSON
            // In the future, this could generate a PDF or image
            return response()->json([
                'status' => true,
                'message' => 'Certification download data',
                'data' => $certification,
                'download_url' => url("/api/development/certifications/{$id}")
            ]);

        } catch (\Exception $e) {
            \Log::error('Certification download API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error downloading certification',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all certifications with user information
     */
    public function getAllWithUsers(Request $request)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Get certifications with user information by joining on id_no = student_id OR govt_id = student_id
            $certifications = Certification::select('certifications.*', 'users.full_name', 'users.email', 'users.mobile')
                ->leftJoin('users', function($join) use ($user) {
                    $join->on('certifications.id_no', '=', 'users.student_id')
                         ->orOn('certifications.govt_id', '=', 'users.student_id');
                })
                ->where(function($query) use ($user) {
                    $query->where('certifications.id_no', $user->student_id)
                          ->orWhere('certifications.govt_id', $user->student_id);
                })
                ->orderBy('certifications.Id', 'desc')
                ->get();

            return response()->json([
                'status' => true,
                'message' => 'Certifications with user data retrieved successfully',
                'data' => $certifications
            ]);

        } catch (\Exception $e) {
            \Log::error('Certifications with users API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error retrieving certifications with user data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get certification by student ID (for admin purposes)
     */
    public function getByStudentId($studentId)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Get certifications for a specific student ID (checking both id_no and govt_id)
            $certifications = Certification::where(function($query) use ($studentId) {
                $query->where('id_no', $studentId)
                      ->orWhere('govt_id', $studentId);
            })
            ->orderBy('Id', 'desc')
            ->get();

            return response()->json([
                'status' => true,
                'message' => 'Certifications retrieved successfully',
                'data' => $certifications
            ]);

        } catch (\Exception $e) {
            \Log::error('Certifications by student ID API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error retrieving certifications by student ID',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get certifications by government ID
     */
    public function getByGovtId($govtId)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Get certifications for a specific government ID
            $certifications = Certification::where('govt_id', $govtId)
                ->orderBy('Id', 'desc')
                ->get();

            return response()->json([
                'status' => true,
                'message' => 'Certifications retrieved successfully',
                'data' => $certifications
            ]);

        } catch (\Exception $e) {
            \Log::error('Certifications by government ID API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error retrieving certifications by government ID',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get certifications by any ID (student_id or govt_id)
     */
    public function getByAnyId($anyId)
    {
        try {
            $user = apiAuth();
            
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'message' => 'User not authenticated'
                ], 401);
            }

            // Get certifications for any ID (checking both id_no and govt_id)
            $certifications = Certification::where(function($query) use ($anyId) {
                $query->where('id_no', $anyId)
                      ->orWhere('govt_id', $anyId);
            })
            ->orderBy('Id', 'desc')
            ->get();

            return response()->json([
                'status' => true,
                'message' => 'Certifications retrieved successfully',
                'data' => $certifications
            ]);

        } catch (\Exception $e) {
            \Log::error('Certifications by any ID API error: ' . $e->getMessage());
            
            return response()->json([
                'status' => false,
                'message' => 'Error retrieving certifications by any ID',
                'error' => $e->getMessage()
            ], 500);
        }
    }
} 