<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Panel\CertificateController;
use Illuminate\Http\Request;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider within a group which
| is assigned the "api" middleware group. Enjoy building your API!
|
*/


// Test email route (temporary for debugging)
Route::get('/test-email', function () {
    try {
        \Log::info('Testing email configuration');
        
        $generalSettings = getGeneralSettings();
        \Log::info('General settings:', $generalSettings);
        
        // Test basic mail configuration
        $mailConfig = [
            'driver' => config('mail.default'),
            'host' => config('mail.mailers.smtp.host'),
            'port' => config('mail.mailers.smtp.port'),
            'encryption' => config('mail.mailers.smtp.encryption'),
            'from_address' => config('mail.from.address'),
            'from_name' => config('mail.from.name'),
        ];
        
        \Log::info('Mail configuration:', $mailConfig);
        
        return response()->json([
            'success' => true,
            'message' => 'Email configuration logged',
            'mail_config' => $mailConfig,
            'general_settings' => $generalSettings
        ]);
    } catch (\Exception $e) {
        \Log::error('Email test failed:', ['error' => $e->getMessage()]);
        return response()->json(['success' => false, 'error' => $e->getMessage()]);
    }
});

Route::group(['prefix' => '/development'], function () {

    // Certificate image saving route
    Route::post('/certificates/save-image', [CertificateController::class, 'saveImage'])->name('save.certificate.image');        

    // Mobile app certificate routes
    Route::group(['prefix' => 'certificates', 'namespace' => 'Web', 'middleware' => 'api.auth'], function () {
        Route::get('/generate-image', 'CertificatesController@generateCertificateImage');
        Route::get('/generate-image-enhanced', 'CertificatesController@generateCertificateImageEnhanced');
        Route::get('/get-html', 'CertificatesController@getCertificateHtml');
        Route::get('/list', 'CertificatesController@getCertificates');
        Route::get('/test', 'CertificatesController@testCertificateGeneration');
    });

    // Certifications routes
    Route::group(['prefix' => 'certifications', 'namespace' => 'Web', 'middleware' => 'api.auth'], function () {
        Route::get('/', 'CertificationsController@index');
        Route::get('/{id}', 'CertificationsController@show');
        Route::get('/{id}/download', 'CertificationsController@download');
        Route::get('/with-users', 'CertificationsController@getAllWithUsers');
        Route::get('/by-student/{studentId}', 'CertificationsController@getByStudentId');
        Route::get('/by-govt-id/{govtId}', 'CertificationsController@getByAnyId');
        Route::get('/by-any-id/{anyId}', 'CertificationsController@getByAnyId');
    });

    // Public certificate test route (no authentication required)
    Route::group(['prefix' => 'certificates', 'namespace' => 'Web'], function () {
        Route::get('/test-public', 'CertificatesController@testCertificateGeneration');
        Route::get('/generate-public', 'CertificatesController@generateCertificateImageEnhanced');
        Route::get('/generate-on-demand', 'CertificatesController@generateCertificateOnDemand');
        Route::get('/generate-simple', 'CertificatesController@generateCertificatePublic');
        Route::get('/generate-enhanced', 'CertificatesController@generateCertificateEnhanced');
        Route::get('/get-with-offline-data', 'CertificatesController@getCertificateWithOfflineData');
        Route::get('/list', 'CertificatesController@getCertificatesList');
    });

    // Simple test route for certificate generation
    Route::get('/test-certificate', function () {
        return response()->json([
            'status' => true,
            'message' => 'Certificate API is working',
            'timestamp' => now()
        ]);
    });

    // Test certificate generation without authentication
    Route::get('/test-certificate-generation', function (Request $request) {
        $certificateId = $request->input('certificate_id');
        $userId = $request->input('user_id');
        $type = $request->input('type', 'quiz');
        
        return response()->json([
            'status' => true,
            'message' => 'Certificate generation test',
            'data' => [
                'certificate_id' => $certificateId,
                'user_id' => $userId,
                'type' => $type,
                'test_url' => url("api/development/certificates/generate-public?certificate_id=$certificateId&user_id=$userId&type=$type")
            ]
        ]);
    });

    Route::get('/', function () {
        return 'api test';
    });

    Route::middleware('api') ->group(base_path('routes/api/auth.php'));

    Route::namespace('Web')->group(base_path('routes/api/guest.php'));

    Route::prefix('panel')->namespace('Panel')->group(base_path('routes/api/user.php'));

    Route::group(['namespace' => 'Config', 'middleware' => []], function () {
        Route::get('/config', ['uses' => 'ConfigController@list']);
        Route::get('/config/register/{type}', ['uses' => 'ConfigController@getRegisterConfig']);
    });

    Route::prefix('instructor')->middleware(['api.auth', 'api.level-access:teacher'])->namespace('Instructor')->group(base_path('routes/api/instructor.php'));

});
