<?php
declare(strict_types=1);

namespace MyOnlineStore\Tests\Omnipay\KlarnaCheckout\Message;

use Money\Currency;
use Money\Money;
use MyOnlineStore\Omnipay\KlarnaCheckout\ItemBag;
use MyOnlineStore\Omnipay\KlarnaCheckout\Message\AbstractRequest;
use Omnipay\Tests\TestCase;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;

final class AbstractRequestTest extends TestCase
{
    /** @var AbstractRequest */
    private $request;

    protected function setUp(): void
    {
        $httpClient = $this->getHttpClient();
        $httpRequest = $this->getHttpRequest();

        $this->request = new class ($httpClient, $httpRequest) extends AbstractRequest
        {
            /**
             * @inheritdoc
             */
            public function sendData($data)
            {
                return [];
            }

            /**
             * @inheritdoc
             */
            public function getData()
            {
                return [];
            }
        };
    }

    public function testGetAmountWithDecimalStringShouldReturnCorrectValue()
    {
        $amount = '5.25';
        $currencyIso = 'EUR';

        $this->request->setCurrency($currencyIso);
        $this->request->setAmount($amount);

        self::assertSame('525', $this->request->getAmount()->getAmount());
    }

    public function testGetResponseBodyWillReturnArray()
    {
        $httpClient = $this->getHttpClient();
        $httpRequest = $this->getHttpRequest();

        $request = new class ($httpClient, $httpRequest) extends AbstractRequest
        {
            public function sendData($data)
            {
                return [];
            }

            public function getData()
            {
                return [];
            }

            // phpcs:disable Generic.CodeAnalysis.UselessOverridingMethod
            public function getResponseBody(ResponseInterface $response): array
            {
                return parent::getResponseBody($response); // TODO: Change the autogenerated stub
            }
        };

        $response = $this->createMock(ResponseInterface::class);
        $stream = $this->createMock(StreamInterface::class);

        $response->expects(self::once())
            ->method('getBody')
            ->willReturn($stream);

        $decodedResponse = ['foo' => 'bar'];
        $stream->expects(self::once())
            ->method('getContents')
            ->willReturn(\json_encode($decodedResponse));

        self::assertSame($decodedResponse, $request->getResponseBody($response));
    }

    public function testGetResponseBodyWillReturnArrayIfResponseIsEmtpy()
    {
        $httpClient = $this->getHttpClient();
        $httpRequest = $this->getHttpRequest();

        $request = new class ($httpClient, $httpRequest) extends AbstractRequest
        {
            public function sendData($data)
            {
                return [];
            }

            public function getData()
            {
                return [];
            }

            public function getResponseBody(ResponseInterface $response): array
            {
                return parent::getResponseBody($response); // TODO: Change the autogenerated stub
            }
        };

        $response = $this->createMock(ResponseInterface::class);
        $stream = $this->createMock(StreamInterface::class);

        $response->expects(self::once())
            ->method('getBody')
            ->willReturn($stream);

        $stream->expects(self::once())
            ->method('getContents')
            ->willReturn(null);

        self::assertSame([], $request->getResponseBody($response));
    }

    public function testGetTaxAmountWithDecimalStringShouldReturnCorrectValue()
    {
        $taxAmount = '5.25';
        $currencyIso = 'EUR';

        $this->request->setCurrency($currencyIso);
        $this->request->setTaxAmount($taxAmount);

        self::assertSame('525', $this->request->getTaxAmount()->getAmount());
    }

    public function testGetters()
    {
        $locale = 'nl_NL';
        $taxAmount = 500;
        $currencyIso = 'EUR';

        $this->request->setCurrency($currencyIso);
        $this->request->setLocale($locale);
        $this->request->setTaxAmount(new Money($taxAmount, new Currency($currencyIso)));

        self::assertSame($locale, $this->request->getLocale());
        self::assertSame((string) $taxAmount, $this->request->getTaxAmount()->getAmount());
    }

    public function testSetItemsWithArray()
    {
        $itemsArray = [['tax_rate' => 10]];

        $this->request->setItems($itemsArray);

        self::assertEquals(new ItemBag($itemsArray), $this->request->getParameters()['items']);
    }

    public function testSetItemsWithItemBag()
    {
        $itemBag = $this->createMock(ItemBag::class);

        $this->request->setItems($itemBag);

        self::assertSame($itemBag, $this->request->getParameters()['items']);
    }
}
