<?php

/**
 * TJ Mega Menu Post Type
 *
 * @author 		Theme Junction 
 * @package 	TJCore
 * @version 	1.0.0
 * 
 * Class TJ_Mega_Menu
 */

if (!defined('ABSPATH')) {
  exit; // Exit if accessed directly
}

if (!class_exists('TJ_Mega_Menu')) {

  final class TJ_Mega_Menu
  {

    /**
     * Singleton instance
     *
     * @var TJ_Mega_Menu|null
     */
    private static $instance = null;

    /**
     * Elementor plugin instance
     *
     * @var \Elementor\Plugin|null
     */
    private static $elementor_instance = null;

    /**
     * Get singleton instance
     *
     * @return TJ_Mega_Menu
     */
    public static function instance()
    {
      if (self::$instance === null) {
        self::$instance = new self();
      }
      return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct()
    {
      add_action('init', [$this, 'register_post_type']);
      add_action('elementor/init', [$this, 'enable_elementor_support']);
      add_filter('single_template', [$this, 'override_canvas_template']);
    }

    /**
     * Register custom post type for Mega Menu templates
     */
    public function register_post_type()
    {
      $labels = [
        'name'               => esc_html__('TJ Mega Menu', 'tjcore'),
        'singular_name'      => esc_html__('Mega Menu', 'tjcore'),
        'menu_name'          => esc_html__('TJ Mega Menu', 'tjcore'),
        'add_new'            => esc_html__('Add New', 'tjcore'),
        'add_new_item'       => esc_html__('Add New Mega Menu', 'tjcore'),
        'edit_item'          => esc_html__('Edit Mega Menu', 'tjcore'),
        'new_item'           => esc_html__('New Mega Menu', 'tjcore'),
        'view_item'          => esc_html__('View Mega Menu', 'tjcore'),
        'search_items'       => esc_html__('Search Mega Menus', 'tjcore'),
        'not_found'          => esc_html__('No Mega Menus found', 'tjcore'),
        'not_found_in_trash' => esc_html__('No Mega Menus found in Trash', 'tjcore'),
      ];

      $args = [
        'labels'             => $labels,
        'public'             => true,
        'show_ui'            => true,
        'show_in_menu'       => true,
        'menu_position'      => 3,
        'menu_icon'          => 'dashicons-screenoptions',
        'supports'           => ['title', 'elementor'],
        'capability_type'    => 'post',
        'has_archive'        => false,
        'rewrite'            => false,
        'show_in_rest'       => true,
      ];

      register_post_type('tj-mega-menu', $args);
    }

    /**
     * Enable Elementor support for Mega Menu post type
     */
    public function enable_elementor_support()
    {
      if (
        defined('ELEMENTOR_VERSION') &&
        version_compare(ELEMENTOR_VERSION, '3.5.0', '>=')
      ) {
        self::$elementor_instance = \Elementor\Plugin::instance();

        if (post_type_exists('tj-mega-menu')) {
          add_post_type_support('tj-mega-menu', 'elementor');
        }
      } else {
        add_action('admin_notices', [$this, 'elementor_required_notice']);
      }
    }

    /**
     * Show admin notice if Elementor is missing or outdated
     */
    public function elementor_required_notice()
    {
      echo '<div class="notice notice-error"><p>';
      esc_html_e('TJ Mega Menu Builder requires Elementor version 3.5.0 or higher.', 'tjcore');
      echo '</p></div>';
    }

    /**
     * Override single template with Elementor Canvas for Mega Menu post type
     *
     * @param string $template
     * @return string
     */
    public function override_canvas_template($template)
    {
      global $post;

      if (
        isset($post->post_type) &&
        $post->post_type === 'tj-mega-menu'
      ) {
        return ELEMENTOR_PATH . '/modules/page-templates/templates/canvas.php';
      }

      return $template;
    }

    /**
     * Utility: Get published post IDs for a given post type
     *
     * @param string $post_type
     * @return array
     */
    public static function get_posts($post_type)
    {
      $args = [
        'post_type'      => $post_type,
        'post_status'    => 'publish',
        'posts_per_page' => -1,
        'orderby'        => 'ID',
        'order'          => 'DESC',
        'fields'         => 'ids',
      ];
      return get_posts($args);
    }
  }

  // Initialize the plugin
  TJ_Mega_Menu::instance();
}


/**
 * Mega Menu Custom Fields
 */
function tj_mega_menu_custom_fields($item_id, $item)
{
  $menu_item_elementor_template = get_post_meta($item_id, '_menu_item_elementor_template', true);
  $menu_item_width = get_post_meta($item_id, '_menu_item_width', true);
  $menu_item_switch = get_post_meta($item_id, '_menu_item_switch', true);
  $custom_class = get_post_meta($item_id, '_menu_item_layout', true); // renamed field
  $checked = $menu_item_switch ? 'checked' : '';

  $post_args = [
    'post_status' => 'publish',
    'post_type' => 'tj-mega-menu',
    'posts_per_page' => -1,
  ];
  $pro_query = new WP_Query($post_args);
?>

  <!-- Enable Switch -->
  <p class="description description-wide">
    <label for="edit-menu-item-switch-<?php echo esc_attr($item_id); ?>">
      <?php _e('Enable Megamenu'); ?>
      <label class="tj-switch">
        <input type="checkbox" class="tj-megamenu-toggle" id="edit-menu-item-switch-<?php echo esc_attr($item_id); ?>" name="menu-item-switch[<?php echo esc_attr($item_id); ?>]" value="1" <?php echo $checked; ?>>
        <span class="slider round"></span>
      </label>
    </label>
  </p>

  <!-- Elementor Template -->
  <div class="tj_megamenu_options_container" style="display: <?php echo $checked ? 'block' : 'none'; ?>">
    <p class="description description-wide">
      <label for="menu_item_elementor_template-<?php echo esc_attr($item_id); ?>">
        <?php esc_html_e('Mega Menu Template', 'tjcore'); ?><br />
        <select class="widefat" id="menu_item_elementor_template-<?php echo esc_attr($item_id); ?>" name="menu_item_elementor_template[<?php echo esc_attr($item_id); ?>]">
          <option value="-1"><?php echo esc_html__('Select A Menu', 'tjcore') ?></option>
          <?php while ($pro_query->have_posts()) : $pro_query->the_post(); ?>
            <?php $selected = ($menu_item_elementor_template == get_the_ID()) ? "selected='selected'" : ''; ?>
            <option value="<?php echo esc_attr(get_the_ID()) ?>" <?php echo $selected; ?>><?php echo esc_html(get_the_title()) ?></option>
          <?php endwhile; ?>
        </select>
      </label>
    </p>

    <!-- Custom Class Input -->
    <p class="description description-wide tj-megamenu-layout" style="display: <?php echo $checked ? 'block' : 'none'; ?>">
      <label for="edit-menu-item-layout-<?php echo esc_attr($item_id); ?>">
        <?php _e('Megamenu Custom Class'); ?><br>
        <input class="widefat" type="text" id="edit-menu-item-layout-<?php echo esc_attr($item_id); ?>"
          name="menu-item-layout[<?php echo esc_attr($item_id); ?>]"
          value="<?php echo esc_attr($custom_class); ?>"
          placeholder="<?php esc_attr_e('type custom class name', 'tjcore'); ?>" />
      </label>
    </p>
    <!-- Custom width -->
    <p class="description description-wide">
      <label for="menu_item_width-<?php echo esc_attr($item_id); ?>">
        <?php esc_html_e('Megamenu Width (px)', 'tjcore'); ?><br />
        <input class="widefat" type="text" name="menu_item_width[<?php echo esc_attr($item_id); ?>]" id="menu_item_width-<?php echo esc_attr($item_id); ?>" value="<?php echo esc_attr($menu_item_width); ?>" placeholder="<?php esc_attr_e('Type only number here', 'tjcore'); ?>" />
      </label>
    </p>
  </div>

<?php
  wp_reset_postdata();
}
add_action('wp_nav_menu_item_custom_fields', 'tj_mega_menu_custom_fields', 10, 2);

/**
 * Save Mega Menu Fields
 */
function tj_mega_menu_custom_fields_save($menu_id, $menu_item_db_id)
{
  // Switch
  if (isset($_POST['menu-item-switch'][$menu_item_db_id])) {
    update_post_meta($menu_item_db_id, '_menu_item_switch', 1);
  } else {
    delete_post_meta($menu_item_db_id, '_menu_item_switch');
  }

  // Custom Class
  if (isset($_POST['menu-item-layout'][$menu_item_db_id])) {
    $layout = sanitize_text_field($_POST['menu-item-layout'][$menu_item_db_id]);
    update_post_meta($menu_item_db_id, '_menu_item_layout', $layout);
  } else {
    delete_post_meta($menu_item_db_id, '_menu_item_layout');
  }

  // Width
  if (isset($_POST['menu_item_width'][$menu_item_db_id])) {
    $width = sanitize_text_field($_POST['menu_item_width'][$menu_item_db_id]);
    update_post_meta($menu_item_db_id, '_menu_item_width', $width);
  } else {
    delete_post_meta($menu_item_db_id, '_menu_item_width');
  }

  // Template
  if (isset($_POST['menu_item_elementor_template'][$menu_item_db_id])) {
    $template_id = absint($_POST['menu_item_elementor_template'][$menu_item_db_id]);
    update_post_meta($menu_item_db_id, '_menu_item_elementor_template', $template_id);
  } else {
    delete_post_meta($menu_item_db_id, '_menu_item_elementor_template');
  }
}
add_action('wp_update_nav_menu_item', 'tj_mega_menu_custom_fields_save', 10, 2);

/**
 * Add Meta to Menu Item Object
 */
function tj_mega_menu_load_meta($items)
{
  foreach ($items as $item) {
    $item->custom_switch = get_post_meta($item->ID, '_menu_item_switch', true);
    $item->megamenu_layout = get_post_meta($item->ID, '_menu_item_layout', true);
  }
  return $items;
}
add_filter('wp_get_nav_menu_items', 'tj_mega_menu_load_meta');


/**
 * Admin JS to Toggle Fields
 */
add_action('admin_footer-nav-menus.php', function () {
?>
  <script>
    jQuery(document).ready(function($) {
      function toggleFields() {
        $('.menu-item').each(function() {
          const $checkbox = $(this).find('.tj-megamenu-toggle');
          const isChecked = $checkbox.is(':checked');
          $(this).find('.tj-megamenu-layout').toggle(isChecked);
          $(this).find('.tj_megamenu_options_container').toggle(isChecked);
        });
      }
      toggleFields();
      $(document).on('change', '.tj-megamenu-toggle', function() {
        toggleFields();
      });
    });
  </script>
  <style>
    .tj-switch {
      display: inline-block;
      position: relative;
      width: 46px;
      height: 24px;
    }

    .tj-switch input {
      opacity: 0;
      width: 0;
      height: 0;
    }

    .slider {
      position: absolute;
      cursor: pointer;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background-color: #ccc;
      transition: .4s;
      border-radius: 24px;
    }

    .slider:before {
      position: absolute;
      content: "";
      height: 18px;
      width: 18px;
      left: 3px;
      bottom: 3px;
      background-color: white;
      transition: .4s;
      border-radius: 50%;
    }

    input:checked+.slider {
      background-color: #007cba;
    }

    input:checked+.slider:before {
      transform: translateX(22px);
    }
  </style>
<?php
});
