<?php

namespace TJCore;

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use \Elementor\Utils;

if (!defined('ABSPATH')) {
  exit;
}
// Exit if accessed directly

/**
 * Elementor widget for Image.
 */
class TJ_Image_Gallery extends Widget_Base
{

  // widget name
  public function get_name()
  {
    return 'tj-image-gallery';
  }

  // widget title.
  public function get_title()
  {
    return __('TJ Image Gallery', 'tjcore');
  }

  // widget icon.
  public function get_icon()
  {
    return 'eicon-image tj-icon';
  }

  // Widget categories.
  public function get_categories()
  {
    return ['tjcore'];
  }

  // scripts dependencies.
  public function get_script_depends()
  {
    return ['tjcore-script'];
  }

  // styles dependencies.
  public function get_style_depends()
  {
    return ['tjcore-style'];
  }

  // Widget keywords.
  public function get_keywords()
  {
    return [
      'image',
      'media',
      'tj image',
      'tj',
      'tj addons',
      'tjcore',
    ];
  }

  // Widget help url.
  public function get_custom_help_url()
  {
    return 'https://go.elementor.com/';
  }

  /**
   * widget controls.
   */
  protected function register_controls()
  {

    // _tj_image_content
    $this->start_controls_section(
      '_tj_image_content',
      [
        'label' => esc_html__('Image', 'tjcore'),
      ]
    );
    $this->add_control(
      'tj_image',
      [
        'label'   => esc_html__('Choose Image', 'tjcore'),
        'type'    => Controls_Manager::MEDIA,
        'default' => [
          'url' => Utils::get_placeholder_image_src(),
        ],
      ]
    );

    $this->add_control(
      'tj_animation',
      [
        'label'        => esc_html__('Enable Animation?', 'tjcore'),
        'type'         => Controls_Manager::SWITCHER,
        'label_on'     => esc_html__('Yes', 'tjcore'),
        'label_off'    => esc_html__('No', 'tjcore'),
        'return_value' => 'yes',
        'default'      => '0',
        'separator'    => 'before',
      ]
    );
    $this->add_control(
      'tj_animation_name',
      [
        'label'     => esc_html__('Animation Name', 'tjcore'),
        'type'      => Controls_Manager::SELECT,
        'default'   => '',
        'options'   => [
          ''            => esc_html__('Default', 'tjcore'),
          'zoomIn'      => esc_html__('Zoom In', 'tjcore'),
          'fadeInUp'    => esc_html__('Fade In Up', 'tjcore'),
          'fadeInDown'  => esc_html__('Fade In Down', 'tjcore'),
          'fadeInLeft'  => esc_html__('Fade In Left', 'tjcore'),
          'fadeInRight' => esc_html__('Fade In Right', 'tjcore'),
        ],
        'condition' => [
          'tj_animation' => 'yes',
        ],
      ]
    );
    $this->add_control(
      'tj_animation_delay',
      [
        'label'      => esc_html__('Animation Delay', 'tjcore'),
        'type'       => Controls_Manager::SLIDER,
        'size_units' => ['px'],
        'range'      => [
          'px' => [
            'min'  => 0,
            'max'  => 3,
            'step' => .1,
          ],
        ],
        'default'    => [
          'unit' => 'px',
          'size' => 0.3,
        ],
        'condition'  => [
          'tj_animation' => 'yes',
        ],
      ]
    );
    $this->end_controls_section();

    /** ---------------------------------------------------------------
     * TAB_STYLE
     * -------------------------------------------------------------* */
    $this->start_controls_section(
      '_section_style_image',
      [
        'label' => esc_html__('Image', 'tjcore'),
        'tab'   => Controls_Manager::TAB_STYLE,
      ]
    );
    $this->add_responsive_control(
      'tj_image_width',
      [
        'label'      => esc_html__('Width', 'tjcore'),
        'type'       => Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'em', 'rem', 'custom'],
        'range'      => [
          'px' => [
            'min'  => 0,
            'max'  => 2000,
            'step' => 1,
          ],
          '%'  => [
            'min' => 0,
            'max' => 100,
          ],
        ],
        'default'    => [
          'unit' => '%',
          'size' => '',
        ],
        'selectors'  => [
          '{{WRAPPER}} .tj-image img' => 'width: {{SIZE}}{{UNIT}};',
        ],
      ]
    );
    $this->add_responsive_control(
      'tj_image_max_width',
      [
        'label'      => esc_html__('Max Width', 'tjcore'),
        'type'       => Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'em', 'rem', 'custom'],
        'range'      => [
          'px' => [
            'min'  => 0,
            'max'  => 2000,
            'step' => 1,
          ],
          '%'  => [
            'min' => 0,
            'max' => 100,
          ],
        ],
        'default'    => [
          'unit' => '%',
          'size' => '',
        ],
        'selectors'  => [
          '{{WRAPPER}} .tj-image img' => 'max-width: {{SIZE}}{{UNIT}};',
        ],
      ]
    );
    $this->add_responsive_control(
      'tj_image_height',
      [
        'label'      => esc_html__('Height', 'tjcore'),
        'type'       => Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'em', 'rem', 'custom'],
        'range'      => [
          'px' => [
            'min'  => 0,
            'max'  => 2000,
            'step' => 1,
          ],
          '%'  => [
            'min' => 0,
            'max' => 100,
          ],
        ],
        'default'    => [
          'unit' => '%',
          'size' => '',
        ],
        'selectors'  => [
          '{{WRAPPER}} .tj-image img' => 'height: {{SIZE}}{{UNIT}};',
        ],
      ]
    );

    $this->add_responsive_control(
      'object-fit',
      [
        'label' => esc_html__('Object Fit', 'tjcore'),
        'type' => Controls_Manager::SELECT,
        'condition' => [
          'tj_image_height[size]!' => '',
        ],
        'options' => [
          '' => esc_html__('Default', 'tjcore'),
          'fill' => esc_html__('Fill', 'tjcore'),
          'cover' => esc_html__('Cover', 'tjcore'),
          'contain' => esc_html__('Contain', 'tjcore'),
          'scale-down' => esc_html__('Scale Down', 'tjcore'),
        ],
        'default' => '',
        'selectors' => [
          '{{WRAPPER}} .tj-image img' => 'object-fit: {{VALUE}};',
        ],
      ]
    );

    $this->add_responsive_control(
      'object-position',
      [
        'label' => esc_html__('Object Position', 'tjcore'),
        'type' => Controls_Manager::SELECT,
        'options' => [
          'center center' => esc_html__('Center Center', 'tjcore'),
          'center left' => esc_html__('Center Left', 'tjcore'),
          'center right' => esc_html__('Center Right', 'tjcore'),
          'top center' => esc_html__('Top Center', 'tjcore'),
          'top left' => esc_html__('Top Left', 'tjcore'),
          'top right' => esc_html__('Top Right', 'tjcore'),
          'bottom center' => esc_html__('Bottom Center', 'tjcore'),
          'bottom left' => esc_html__('Bottom Left', 'tjcore'),
          'bottom right' => esc_html__('Bottom Right', 'tjcore'),
        ],
        'default' => 'center center',
        'selectors' => [
          '{{WRAPPER}} .tj-image img' => 'object-position: {{VALUE}};',
        ],
        'condition' => [
          'tj_image_height[size]!' => '',
          'object-fit' => ['cover', 'contain', 'scale-down'],
        ],
      ]
    );

    $this->add_control(
      'tj_image_overlay',
      [
        'label'     => esc_html__('Color Overlay', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'default'   => '',
        'selectors' => [
          '{{WRAPPER}} .tj-image::before' => 'background: {{VALUE}}',
        ],
      ]
    );

    $this->add_control(
      'color_opacity',
      [
        'label' => esc_html__('Color Opacity', 'elementor'),
        'type' => Controls_Manager::SLIDER,
        'range' => [
          'px' => [
            'max' => 1,
            'min' => 0,
            'step' => 0.01,
          ],
        ],
        'selectors' => [
          '{{WRAPPER}} .tj-image::before' => 'opacity: {{SIZE}};',
        ],
      ]
    );

    $this->add_group_control(
      \Elementor\Group_Control_Border::get_type(),
      [
        'name' => 'tj_image_border',
        'selector' => '{{WRAPPER}} .tj-image',
      ]
    );

    $this->add_responsive_control(
      'tj_image_border_radius',
      [
        'label'      => esc_html__('Border Radius', 'tjcore'),
        'type'       => Controls_Manager::DIMENSIONS,
        'size_units' => ['px', '%', 'em', 'rem', 'custom'],
        'selectors'  => [
          '{{WRAPPER}} .tj-image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
          '{{WRAPPER}} .tj-image img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
      ]
    );
    $this->end_controls_section();
  }

  /**
   * Render the widget output on the frontend.
   */
  protected function render()
  {

    $settings = $this->get_settings_for_display();

    // animation
    $tjAnimation      = $settings['tj_animation'] ?? '';
    $tjAnimationName  = $settings['tj_animation_name'] ?? '';
    $tjAnimationDelay = $settings['tj_animation_delay'] ?? '';

    if (!empty($tjAnimation)) {
      $imageClass = "image-box tj-gallery-item mb-0 wow $tjAnimationName";
      $this->add_render_attribute('tj-image-arg', 'data-wow-delay', $tjAnimationDelay['size'] . "s");
    } else {
      $imageClass = "image-box tj-gallery-item mb-0";
    }
    $this->add_render_attribute('tj-image-arg', 'class', $imageClass);

?>

    <div <?php echo $this->get_render_attribute_string('tj-image-arg'); ?>>
      <?php if (!empty($settings['tj_image']['url'])): ?>
        <a class="gallery-popup tj-image" href="<?php echo esc_url($settings['tj_image']['url']); ?>">
          <img src="<?php echo esc_url($settings['tj_image']['url']); ?>" alt="<?php echo esc_attr(get_post_meta($settings['tj_image']['id'], '_wp_attachment_image_alt', true)); ?>">
        </a>
      <?php endif; ?>
    </div>

<?php

  }
}
$widgets_manager->register(new TJ_Image_Gallery());
