<?php

namespace TJCore;

use Elementor\Controls_Manager;
use Elementor\Widget_Base;

if (! defined('ABSPATH')) {
  exit; // Exit if accessed directly
}

/**
 * Elementor Widget: TJ Project Filter
 */
class TJ_Project_Filter extends Widget_Base
{

  use \TJCore\TJCore_Elelmentor_Trait;

  public function get_name()
  {
    return 'tj-project-filter';
  }

  public function get_title()
  {
    return __('TJ Project Filter', 'tjcore');
  }

  public function get_icon()
  {
    return 'eicon-post tj-icon';
  }

  public function get_categories()
  {
    return ['tjcore'];
  }

  public function get_script_depends()
  {
    return ['tjcore-script'];
  }

  public function get_style_depends()
  {
    return ['tjcore-style'];
  }

  public function get_keywords()
  {
    return ['project', 'projects', 'tj project', 'tj', 'tj addons', 'tjcore'];
  }

  public function get_custom_help_url()
  {
    return 'https://go.elementor.com/';
  }

  /**
   * Register widget controls
   */
  protected function register_controls()
  {

    $taxonomy = 'projects-cat';

    // Project Query Section
    $this->start_controls_section(
      'tj_project_query',
      [
        'label' => esc_html__('Project Query', 'tjcore'),
      ]
    );

    $this->add_control(
      'posts_per_page',
      [
        'label'       => esc_html__('Projects Per Page', 'tjcore'),
        'description' => esc_html__('Leave blank or enter -1 for all.', 'tjcore'),
        'type'        => Controls_Manager::NUMBER,
        'default'     => 6,
      ]
    );

    $this->add_control(
      'category',
      [
        'label'       => esc_html__('Include Categories', 'tjcore'),
        'description' => esc_html__('Select categories to include or leave blank for all.', 'tjcore'),
        'type'        => Controls_Manager::SELECT2,
        'multiple'    => true,
        'options'     => tj_get_categories($taxonomy),
        'label_block' => true,
      ]
    );

    $this->add_control(
      'tj_post_title_word',
      [
        'label'       => esc_html__('Title Word Limit', 'tjcore'),
        'description' => esc_html__('Set how many words you want to display.', 'tjcore'),
        'type'        => Controls_Manager::NUMBER,
        'default'     => 6,
      ]
    );

    $this->add_control(
      'tj_category',
      [
        'label'        => esc_html__('Enable Category?', 'tjcore'),
        'type'         => Controls_Manager::SWITCHER,
        'label_on'     => esc_html__('Yes', 'tjcore'),
        'label_off'    => esc_html__('No', 'tjcore'),
        'return_value' => 'yes',
        'default'      => 'yes',
        'separator'    => 'before',
      ]
    );


    $this->add_control(
      'tj_button',
      [
        'label'        => esc_html__('Enable Button?', 'tjcore'),
        'type'         => Controls_Manager::SWITCHER,
        'label_on'     => esc_html__('Yes', 'tjcore'),
        'label_off'    => esc_html__('No', 'tjcore'),
        'return_value' => 'yes',
        'default'      => 'yes',
        'separator'    => 'before',
      ]
    );

    $this->add_control(
      'tj_all_btn',
      [
        'label'       => esc_html__('Filter Button', 'tjcore'),
        'label_block' => true,
        'type'        => Controls_Manager::TEXT,
        'default'     => esc_html__('All', 'tjcore'),
        'condition'   => [
          'tj_button'       => 'yes',
        ],
      ]
    );

    $this->add_control(
      'tj_project_btn',
      [
        'label'       => esc_html__('Button Text', 'tjcore'),
        'label_block' => true,
        'type'        => Controls_Manager::TEXT,
        'default'     => esc_html__('View Project', 'tjcore'),
        'condition'   => [
          'tj_button'       => 'yes',
        ],
      ]
    );

    $this->add_control(
      'tj_pagination',
      [
        'label' => esc_html__('Enable Pagination?', 'tjcore'),
        'type' => Controls_Manager::SWITCHER,
        'label_on' => esc_html__('Yes', 'tjcore'),
        'label_off' => esc_html__('No', 'tjcore'),
        'return_value' => 'yes',
        'default' => '',
        'separator' => 'before',
      ]
    );

    $this->end_controls_section();

    $this->start_controls_section(
      'section_column_settings',
      [
        'label' => esc_html__('Column Settings', 'tjcore'),
        'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
      ]
    );

    $this->add_control(
      'desktop_column',
      [
        'label' => esc_html__('Columns for Desktop', 'tjcore'),
        'description' => esc_html__('Screen width equal to or greater than 1200px', 'tjcore'),
        'type' => \Elementor\Controls_Manager::SELECT,
        'options' => [
          12 => esc_html__('1', 'tjcore'),
          6 => esc_html__('2', 'tjcore'),
          4 => esc_html__('3', 'tjcore'),
          3 => esc_html__('4', 'tjcore'),
          2 => esc_html__('6', 'tjcore'),
        ],
        'separator' => 'before',
        'default' => '6',
        'style_transfer' => true,
      ]
    );

    $this->add_control(
      'laptop_column',
      [
        'label' => esc_html__('Columns for Laptop', 'tjcore'),
        'description' => esc_html__('Screen width equal to or greater than 992px', 'tjcore'),
        'type' => \Elementor\Controls_Manager::SELECT,
        'options' => [
          12 => esc_html__('1', 'tjcore'),
          6 => esc_html__('2', 'tjcore'),
          4 => esc_html__('3', 'tjcore'),
          3 => esc_html__('4', 'tjcore'),
          2 => esc_html__('6', 'tjcore'),
        ],
        'separator' => 'before',
        'default' => '6',
        'style_transfer' => true,
      ]
    );

    $this->add_control(
      'tablet_column',
      [
        'label' => esc_html__('Columns for Tablet', 'tjcore'),
        'description' => esc_html__('Screen width equal to or greater than 768px', 'tjcore'),
        'type' => \Elementor\Controls_Manager::SELECT,
        'options' => [
          12 => esc_html__('1', 'tjcore'),
          6 => esc_html__('2', 'tjcore'),
          4 => esc_html__('3', 'tjcore'),
          3 => esc_html__('4', 'tjcore'),
          2 => esc_html__('6', 'tjcore'),
        ],
        'separator' => 'before',
        'default' => '6',
        'style_transfer' => true,
      ]
    );
    $this->add_control(
      'mobile_column',
      [
        'label' => esc_html__('Columns for Mobile', 'tjcore'),
        'description' => esc_html__('Screen width less than 768px', 'tjcore'),
        'type' => \Elementor\Controls_Manager::SELECT,
        'options' => [
          12 => esc_html__('1', 'tjcore'),
          6 => esc_html__('2', 'tjcore'),
          4 => esc_html__('3', 'tjcore'),
          3 => esc_html__('4', 'tjcore'),
          2 => esc_html__('6', 'tjcore'),
        ],
        'separator' => 'before',
        'default' => '12',
        'style_transfer' => true,
      ]
    );

    $this->end_controls_section();

    /**
     * Style Tabs
     */
    $this->tj_card_style_controls('project_card', 'Card - Style', '.tj-el-project');
    $this->tj_basic_style_controls('project_title', 'Title - Style', '.tj-el-title');
    $this->tj_basic_style_controls('project_cat', 'Category - Style', '.tj-el-cat');


    /**
     * Filter Button Style Controls
     */
    $this->start_controls_section(
      'tj_filter_button_style',
      [
        'label' => esc_html__('Filter Buttons', 'tjcore'),
        'tab'   => Controls_Manager::TAB_STYLE,
      ]
    );

    // Typography
    $this->add_group_control(
      \Elementor\Group_Control_Typography::get_type(),
      [
        'name'     => 'filter_btn_typography',
        'selector' => '{{WRAPPER}} .filter-button-group button',
      ]
    );

    // Normal Style
    $this->start_controls_tabs('filter_btn_tabs');

    $this->start_controls_tab(
      'filter_btn_normal',
      [
        'label' => esc_html__('Normal', 'tjcore'),
      ]
    );

    $this->add_control(
      'filter_btn_text_color',
      [
        'label'     => esc_html__('Text Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .filter-button-group button' => 'color: {{VALUE}};',
        ],
      ]
    );

    $this->add_control(
      'filter_btn_bg_color',
      [
        'label'     => esc_html__('Background Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .filter-button-group button' => 'background-color: {{VALUE}};',
        ],
      ]
    );

    $this->end_controls_tab();

    // Hover/Active Style
    $this->start_controls_tab(
      'filter_btn_hover',
      [
        'label' => esc_html__('Hover / Active', 'tjcore'),
      ]
    );

    $this->add_control(
      'filter_btn_hover_text_color',
      [
        'label'     => esc_html__('Text Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .filter-button-group button:hover, {{WRAPPER}} .filter-button-group button.active' => 'color: {{VALUE}};',
        ],
      ]
    );

    $this->add_control(
      'filter_btn_hover_bg_color',
      [
        'label'     => esc_html__('Background Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .filter-button-group button:hover, {{WRAPPER}} .filter-button-group button.active, {{WRAPPER}} .filter-button-group button.active::after, {{WRAPPER}} .filter-button-group button:hover::after' => 'background-color: {{VALUE}};',
        ],
      ]
    );

    $this->end_controls_tab();

    $this->end_controls_tabs();

    // Border
    $this->add_group_control(
      \Elementor\Group_Control_Border::get_type(),
      [
        'name'     => 'filter_btn_border',
        'selector' => '{{WRAPPER}} .filter-button-group button',
      ]
    );

    // Border radius
    $this->add_control(
      'filter_btn_border_radius',
      [
        'label'      => esc_html__('Border Radius', 'tjcore'),
        'type'       => Controls_Manager::DIMENSIONS,
        'selectors'  => [
          '{{WRAPPER}} .filter-button-group button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
      ]
    );

    // Padding
    $this->add_responsive_control(
      'filter_btn_padding',
      [
        'label'      => esc_html__('Padding', 'tjcore'),
        'type'       => Controls_Manager::DIMENSIONS,
        'size_units' => ['px', '%', 'em'],
        'selectors'  => [
          '{{WRAPPER}} .filter-button-group button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
      ]
    );

    $this->end_controls_section();

    /**
     * Pagination Button Style Controls
     */
    $this->start_controls_section(
      'tj_pagination_button_style',
      [
        'label' => esc_html__('Pagination Style', 'tjcore'),
        'tab'   => Controls_Manager::TAB_STYLE,
      ]
    );

    $this->add_group_control(
      \Elementor\Group_Control_Typography::get_type(),
      [
        'name'     => 'pagination_btn_typography',
        'selector' => '{{WRAPPER}} .tj-pagination ul li .page-numbers',
      ]
    );

    // Normal Style
    $this->start_controls_tabs('pagination_btn_tabs');

    $this->start_controls_tab(
      'pagination_btn_normal',
      [
        'label' => esc_html__('Normal', 'tjcore'),
      ]
    );
    $this->add_control(
      'pagination_btn_text_color',
      [
        'label'     => esc_html__('Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .tj-pagination ul li .page-numbers' => 'color: {{VALUE}};',
        ],
      ]
    );
    $this->add_control(
      'pagination_btn_bg_color',
      [
        'label'     => esc_html__('Background Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .tj-pagination ul li .page-numbers' => 'background-color: {{VALUE}};',
        ],
      ]
    );
    $this->end_controls_tab();

    // Hover/Active Style
    $this->start_controls_tab(
      'pagination_btn_hover',
      [
        'label' => esc_html__('Hover / Active', 'tjcore'),
      ]
    );

    $this->add_control(
      'pagination_btn_hover_text_color',
      [
        'label'     => esc_html__('Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .tj-pagination ul li .page-numbers:hover, {{WRAPPER}} .tj-pagination ul li .page-numbers.current' => 'color: {{VALUE}};',
        ],
      ]
    );

    $this->add_control(
      'pagination_btn_hover_bg_color',
      [
        'label'     => esc_html__('Background Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .tj-pagination ul li .page-numbers:hover, {{WRAPPER}} .tj-pagination ul li .page-numbers.current' => 'background-color: {{VALUE}};',
        ],
      ]
    );

    $this->add_control(
      'pagination_btn_hover_border_color',
      [
        'label'     => esc_html__('Border Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .tj-pagination ul li .page-numbers:hover, {{WRAPPER}} .tj-pagination ul li .page-numbers.current' => 'border-color: {{VALUE}};',
        ],
      ]
    );

    $this->end_controls_tab();
    $this->end_controls_tabs();
    // Border
    $this->add_group_control(
      \Elementor\Group_Control_Border::get_type(),
      [
        'name'     => 'pagination_btn_border',
        'selector' => '{{WRAPPER}} .tj-pagination ul li .page-numbers',
      ]
    );

    // Border radius
    $this->add_control(
      'pagination_btn_border_radius',
      [
        'label'      => esc_html__('Border Radius', 'tjcore'),
        'type'       => Controls_Manager::DIMENSIONS,
        'selectors'  => [
          '{{WRAPPER}} .tj-pagination ul li .page-numbers' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
      ]
    );

    $this->end_controls_section();
  }

  /**
   * Render the widget on the frontend
   */
  protected function render()
  {

    $settings = $this->get_settings_for_display();
    extract($settings);
    $posts_per_page = ! empty($settings['posts_per_page']) ? intval($settings['posts_per_page']) : 6;
    $categories     = ! empty($settings['category']) ? $settings['category'] : [];
    $title_limit    = ! empty($settings['tj_post_title_word']) ? intval($settings['tj_post_title_word']) : 6;

    $paged = get_query_var('paged') ?: (get_query_var('page') ?: 1);

    $args = [
      'paged'          => $paged,
      'post_type'      => 'projects',
      'post_status'    => 'publish',
      'posts_per_page' => $posts_per_page,
    ];

    if (! empty($categories)) {
      $args['tax_query'] = [
        [
          'taxonomy' => 'projects-cat',
          'field'    => 'slug',
          'terms'    => $categories,
        ],
      ];
    }

    $query = new \WP_Query($args);
?>

    <div class="portfolio-filter h10-project-filter text-center wow fadeInUp" data-wow-delay=".3s">
      <div class="button-group h10-project-button-group filter-button-group">
        <?php if (! empty($categories)) : ?>
          <?php if (! empty($settings['tj_all_btn'])) : ?>
            <button class="active" data-filter="*">
              <?php echo esc_html($settings['tj_all_btn']); ?>
            </button>
          <?php endif; ?>

          <?php foreach ($categories as $term_slug) :
            $term = get_term_by('slug', $term_slug, 'projects-cat');
            if (! $term || is_wp_error($term)) {
              continue;
            }
          ?>
            <button data-filter=".<?php echo esc_attr($term_slug); ?>">
              <?php echo esc_html($term->name); ?>
            </button>
          <?php endforeach; ?>
        <?php endif; ?>
        <div class="active-bg"></div>
      </div>
    </div>

    <div class="row grid project-wrapper h7-project-wrapper h10-project-wrapper portfolio-filter-box">

      <?php if ($query->have_posts()) : ?>
        <?php while ($query->have_posts()) : $query->the_post(); ?>

          <?php
          $item_terms = get_the_terms(get_the_ID(), 'projects-cat');
          $class_names = '';

          if ($item_terms && ! is_wp_error($item_terms)) {
            foreach ($item_terms as $item_term) {
              $class_names .= ' ' . sanitize_html_class($item_term->slug);
            }
          }
          ?>
          <div class="col-xl-<?php echo esc_attr($desktop_column); ?> col-lg-<?php echo esc_attr($laptop_column); ?> col-md-<?php echo esc_attr($tablet_column); ?> col-<?php echo esc_attr($mobile_column); ?> <?php echo esc_attr($class_names); ?>">
            <div class="project-item h4-project-item tj-el-project portfolio-filter-item<?php echo esc_attr($class_names); ?>">
              <?php if (has_post_thumbnail()) : ?>
                <div class="project-img"
                  <?php if (! empty($settings['tj_button']) && 'yes' === $settings['tj_button']) : ?>
                  data-cursor-text="<?php echo esc_attr($settings['tj_project_btn']); ?>"
                  <?php endif; ?>>
                  <a href="<?php the_permalink(); ?>">
                    <?php the_post_thumbnail('full'); ?>
                  </a>
                </div>
              <?php endif; ?>

              <div class="project-content">
                <h4 class="title tj-el-title">
                  <a href="<?php the_permalink(); ?>">
                    <?php echo esc_html(wp_trim_words(get_the_title(), $title_limit, '...')); ?>
                  </a>
                </h4>
                <?php if (! empty($settings['tj_category']) && 'yes' === $settings['tj_category'] && $item_terms) : ?>
                  <span class="categories">
                    <?php
                    $first_term = $item_terms[0];
                    $term_link  = get_term_link($first_term);
                    if (! is_wp_error($term_link)) {
                      echo '<a class="tj-el-cat" href="' . esc_url($term_link) . '">' . esc_html($first_term->name) . '</a>';
                    }
                    ?>
                  </span>
                <?php endif; ?>
              </div>
            </div>
          </div>
        <?php endwhile;
        wp_reset_postdata(); ?>
      <?php endif; ?>
    </div>
    <?php if (!empty($settings['tj_pagination'])) : ?>
      <div class="tj-pagination d-flex justify-content-center">
        <?php
        echo paginate_links([
          'base'      => str_replace(999999999, '%#%', esc_url(get_pagenum_link(999999999))),
          'format'    => '?paged=%#%',
          'current'   => $paged,
          'total'     => $query->max_num_pages,
          'prev_text' => '<i class="tji-arrow-left-long"></i>',
          'next_text' => '<i class="tji-arrow-right-long"></i>',
          'type'      => 'list',
          'end_size'  => 2,
          'mid_size'  => 2,
        ]);
        ?>
      </div>
    <?php endif; ?>
<?php
  }

  protected function content_template() {}
}

$widgets_manager->register(new TJ_Project_Filter());
